<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\ORM\Query;
use Cake\ORM\Table;
use Cake\ORM\TableRegistry;

class HomeController extends AppController
{
    private Table $CalendarEvents;
    private Table $Participants;

    public function initialize(): void
    {
        parent::initialize();
        $this->CalendarEvents = $this->fetchTable('CalendarEvents');
        $this->Participants   = $this->fetchTable('Participants');
        $this->viewBuilder()->setLayout('default');
    }

    private function currentClinicIdOrNull(): ?int
    {
        if (method_exists($this, 'currentClinicId')) {
            $v = (int)($this->currentClinicId() ?? 0);
            return $v ?: null;
        }
        $cid = $this->request->getSession()->read('Current.clinic_id');
        return $cid ? (int)$cid : null;
    }

    private function tableExists(string $name): bool
    {
        try {
            $locator = TableRegistry::getTableLocator();
            $table   = $locator->get($name);
            $list    = $table->getConnection()->getSchemaCollection()->listTables();
            return \in_array($table->getTable(), $list, true);
        } catch (\Throwable $e) {
            return false;
        }
    }

    private function scopeEventsToClinic(Query $q, ?int $clinicId): Query
    {
        if (!$clinicId) return $q;

        $events = $this->CalendarEvents;
        $hasParticipantId = $events->hasField('participant_id');
        $hasPCLink        = $this->tableExists('ParticipantsClinics');

        $q->leftJoin(['cu' => 'clinic_users'], 'cu.login_id = ' . $events->aliasField('user_id'));
        $or = [['cu.clinic_id' => $clinicId]];

        if ($hasParticipantId && $hasPCLink) {
            $q->leftJoin(['pc' => 'participants_clinics'], 'pc.participant_id = ' . $events->aliasField('participant_id'));
            $or[] = ['pc.clinic_id' => $clinicId];
        }

        return $q->where(['OR' => $or]);
    }

    private function scopeParticipantsToClinic(Query $q, ?int $clinicId): Query
    {
        if (!$clinicId) return $q;

        if ($this->tableExists('ParticipantsClinics')) {
            $q->innerJoin(
                ['pc' => 'participants_clinics'],
                'pc.participant_id = ' . $this->Participants->aliasField('id')
            )->where(['pc.clinic_id' => $clinicId]);
        }
        return $q;
    }

    private function applySoftDeleteFilter(Query $q, Table $table): Query
    {
        $alias = $table->getAlias();
        $sch   = $table->getSchema();

        if ($sch->hasColumn('deleted')) {
            $q->where([$alias . '.deleted IS' => null]);
        }
        if ($sch->hasColumn('is_deleted')) {
            $q->where([$alias . '.is_deleted' => 0]);
        }
        return $q;
    }


    public function index(): void
    {
        $clinicId = $this->currentClinicIdOrNull();

        $eventQ = $this->CalendarEvents->find();
        $this->scopeEventsToClinic($eventQ, $clinicId);
        $this->applySoftDeleteFilter($eventQ, $this->CalendarEvents);
        $bookingTotal = (int)$eventQ->count();

        $partQ = $this->Participants->find();
        $this->scopeParticipantsToClinic($partQ, $clinicId);
        $this->applySoftDeleteFilter($partQ, $this->Participants);
        $participantTotal = (int)$partQ->count();

        $this->set(compact('bookingTotal', 'participantTotal'));
    }
}
