<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\I18n\FrozenTime;
use Cake\Mailer\Mailer;

class RemindersController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->Reminders = $this->fetchTable('Reminders');
        $this->CalendarEvents = $this->fetchTable('CalendarEvents');
        $this->viewBuilder()->setClassName('Json');
    }

    private function readJson(): array
    {
        $raw = (string)$this->request->getBody();
        $data = json_decode($raw, true);
        return is_array($data) ? $data : (array)$this->request->getData();
    }

    public function schedule()
    {
        $this->request->allowMethod(['post']);
        try {
            $d       = $this->readJson();
            $eventId = (int)($d['event_id'] ?? 0);
            $offset  = (int)($d['offset_minutes'] ?? 0);
            $tpl     = (string)($d['template'] ?? 'default');
            $email   = (string)($d['to_email'] ?? '');

            if ($eventId <= 0) {
                return $this->response->withStatus(422)
                    ->withType('json')->withStringBody(json_encode(['error' => 'event_id required']));
            }

            $event = $this->CalendarEvents->find()
                ->select(['id', 'start'])
                ->where(['id' => $eventId])
                ->first();

            if (!$event || !$event->start) {
                return $this->response->withStatus(404)
                    ->withType('json')->withStringBody(json_encode(['error' => 'event not found']));
            }

            if ($email === '') {
                return $this->response->withType('json')->withStringBody(json_encode(['ok' => true, 'skipped' => true]));
            }

            $sendAt = (new FrozenTime($event->start))->subMinutes($offset);

            $rem = $this->Reminders->newEntity([
                'calendar_event_id' => $event->id,
                'to_email'          => $email,
                'template'          => $tpl ?: 'default',
                'offset_minutes'    => $offset,
                'send_at'           => $sendAt,
                'status'            => 'pending',
            ]);

            if (!$this->Reminders->save($rem)) {
                return $this->response->withStatus(422)
                    ->withType('json')->withStringBody(json_encode(['error' => 'save failed', 'errors' => $rem->getErrors()]));
            }

            return $this->response->withType('json')->withStringBody(json_encode(['ok' => true, 'id' => (string)$rem->id]));
        } catch (\Throwable $e) {
            \Cake\Log\Log::error('Reminder schedule failed: ' . $e->getMessage());
            return $this->response->withStatus(500)
                ->withType('json')->withStringBody(json_encode(['error' => $e->getMessage()]));
        }
    }

    public function cancel($eventId = null)
    {
        $this->request->allowMethod(['post']);
        if (!$eventId) {
            $d = $this->readJson();
            $eventId = (int)($d['event_id'] ?? 0);
        }
        if ($eventId <= 0) {
            return $this->response->withStatus(422)
                ->withType('json')->withStringBody(json_encode(['error' => 'event_id required']));
        }

        $this->Reminders->updateAll(['status' => 'cancelled'], [
            'calendar_event_id' => $eventId,
            'status !=' => 'cancelled'
        ]);

        return $this->response->withType('json')->withStringBody(json_encode(['ok' => true]));
    }

    public function resync($eventId = null)
    {
        $this->request->allowMethod(['post']);
        if (!$eventId) {
            $d = $this->readJson();
            $eventId = (int)($d['event_id'] ?? 0);
        }
        if ($eventId <= 0) {
            return $this->response->withStatus(422)
                ->withType('json')->withStringBody(json_encode(['error' => 'event_id required']));
        }

        $event = $this->CalendarEvents->find()->select(['id', 'start'])->where(['id' => $eventId])->first();
        if (!$event || !$event->start) {
            return $this->response->withStatus(404)
                ->withType('json')->withStringBody(json_encode(['error' => 'event not found']));
        }

        $items = $this->Reminders->find()
            ->where(['calendar_event_id' => $eventId, 'status' => 'pending'])
            ->all();

        foreach ($items as $r) {
            $r->send_at = (new FrozenTime($event->start))->subMinutes((int)$r->offset_minutes);
            $this->Reminders->save($r);
        }

        return $this->response->withType('json')->withStringBody(json_encode(['ok' => true]));
    }

    public function dispatch()
    {
        $this->request->allowMethod(['post', 'get']);

        $now = FrozenTime::now();
        $items = $this->Reminders->find()
            ->where(['status' => 'pending', 'send_at <=' => $now])
            ->orderAsc('send_at')
            ->limit(50)
            ->all();

        $sent = 0; $failed = 0; $errors = [];

        foreach ($items as $r) {
            try {
                $ev = $this->CalendarEvents->find()
                    ->select(['title','start','location'])
                    ->where(['id' => $r->calendar_event_id])
                    ->first();

                $subject = 'Appointment Reminder';
                $body = "This is a reminder for your appointment."
                    . "\nTitle: " . ($ev->title ?? '-')
                    . "\nStart: " . (($ev->start instanceof FrozenTime) ? $ev->start->i18nFormat('yyyy-MM-dd HH:mm') : (string)($ev->start ?? '-'))
                    . "\nLocation: " . ($ev->location ?? '-')
                    . "\n\nSee you soon.";

                (new Mailer('default'))
                    ->setEmailFormat('text')
                    ->setTo($r->to_email)
                    ->setSubject($subject)
                    ->deliver($body);

                $r->status = 'sent';
                $this->Reminders->save($r);
                $sent++;
            } catch (\Throwable $e) {
                $r->status = 'failed';
                $this->Reminders->save($r);
                $failed++;
                $errors[] = $e->getMessage();
            }
        }

        return $this->response->withType('json')
            ->withStringBody(json_encode(['ok' => true, 'sent' => $sent, 'failed' => $failed, 'errors' => $errors]));
    }

    public function sendNow()
    {
        $this->request->allowMethod(['post']);
        $d = $this->readJson();

        $eventId = (int)($d['event_id'] ?? 0);
        $email   = trim((string)($d['to_email'] ?? ''));
        $tpl     = (string)($d['template'] ?? 'default');

        if ($eventId <= 0 || $email === '') {
            return $this->response->withStatus(422)
                ->withType('json')
                ->withStringBody(json_encode(['error' => 'event_id and to_email required']));
        }

        $ev = $this->CalendarEvents->find()
            ->select(['id','title','start','location'])
            ->where(['id' => $eventId])
            ->first();

        if (!$ev) {
            return $this->response->withStatus(404)
                ->withType('json')
                ->withStringBody(json_encode(['error' => 'event not found']));
        }

        try {
            $subject = 'Appointment Reminder';
            $startStr = ($ev->start instanceof FrozenTime)
                ? $ev->start->i18nFormat('yyyy-MM-dd HH:mm')
                : (string)($ev->start ?? '-');

            $body = "This is a reminder for your appointment."
                . "\nTitle: " . ($ev->title ?? '-')
                . "\nStart: " . $startStr
                . "\nLocation: " . ($ev->location ?? '-')
                . "\n\nSee you soon.";

            (new Mailer('default'))
                ->setEmailFormat('text')
                ->setTo($email)
                ->setSubject($subject)
                ->deliver($body);

            $rem = $this->Reminders->newEntity([
                'calendar_event_id' => $ev->id,
                'to_email'          => $email,
                'template'          => $tpl ?: 'default',
                'offset_minutes'    => 0,
                'send_at'           => FrozenTime::now(),
                'status'            => 'sent',
            ]);
            $this->Reminders->save($rem);

            return $this->response->withType('json')
                ->withStringBody(json_encode(['ok' => true]));
        } catch (\Throwable $e) {
            return $this->response->withStatus(500)
                ->withType('json')
                ->withStringBody(json_encode(['error' => $e->getMessage()]));
        }
    }

}
