<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Table;
use Cake\Validation\Validator;
use Cake\ORM\RulesChecker;

class ClinicServicesTable extends Table
{
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('clinic_services');
        $this->setPrimaryKey('id');
        $this->addBehavior('Timestamp');

        $this->belongsTo('Clinics', [
            'foreignKey' => 'clinic_id',
            'joinType'   => 'INNER',
        ]);
    }

    public function validationDefault(Validator $v): Validator
    {
        return $v
            ->integer('clinic_id')->requirePresence('clinic_id')
            ->scalar('name')->maxLength('name', 255)->requirePresence('name')->notEmptyString('name')
            ->scalar('code')->allowEmptyString('code') // optional, but unique per clinic if provided
            ->integer('duration_minutes')->requirePresence('duration_minutes')->notEmptyString('duration_minutes')
            ->integer('price_cents')->allowEmptyString('price_cents')
            ->boolean('active')->allowEmptyString('active');
    }

    public function buildRules(RulesChecker $rules): RulesChecker
    {
        // unique (clinic_id, code) when code not null
        $rules->add(function ($entity, $options) {
            if ($entity->code === null || $entity->code === '') return true;
            $exists = $this->exists([
                'clinic_id' => $entity->clinic_id,
                'code'      => $entity->code,
                'id !='     => $entity->id ?? 0,
            ]);
            return !$exists;
        }, 'uniqCodePerClinic', [
            'errorField' => 'code',
            'message'    => 'Code must be unique within this clinic.'
        ]);

        return $rules;
    }
}
