(function () {
    'use strict';

    function __resolveServiceLabel(xp) {
        if (!xp || typeof xp !== 'object') return '';
        const candidates = [
            xp.service_type, xp.serviceType,
            xp.service_name, xp.serviceName,
            xp.service, xp.service_label, xp.serviceLabel,
            xp.service_code, xp.serviceCode
        ];
        for (const v of candidates) {
            const t = (v == null ? '' : String(v)).replace(/^name:/,'').trim();
            if (t) return t;
        }
        const id = (xp.service_id != null ? xp.service_id : xp.serviceId);
        if (id != null) {
            try {
                if (typeof window.__svcNameById === 'function') {
                    const byMap = window.__svcNameById(id);
                    if (byMap) return byMap;
                }
                const row = (window.__svcAll || []).find(r => String(r.id) === String(id));
                if (row && row.name) return row.name;
            } catch (_) {}
        }
        return '';
    }

    window.__svcNameMap = (() => {
        try {
            const list = JSON.parse(localStorage.getItem('cal__services_v1') || '[]') || [];
            const map = {};
            list.forEach(x => { if (x && x.id != null && x.name) map[String(x.id)] = String(x.name); });
            return map;
        } catch (_) { return {}; }
    })();

    window.__svcNameById = function (id) {
        const key = String(id);
        if (window.__svcNameMap && window.__svcNameMap[key]) return window.__svcNameMap[key];
        const row = (window.__svcAll || []).find(x => String(x.id) === key);
        return row ? String(row.name || '') : '';
    };

    const cfg = window.__CAL_CFG__ || {};
    const $ = window.jQuery;
    const ADD_OPT_PREFIX = '＋ Add: ';

    const buildAddOptValue = (name) => `${ADD_OPT_PREFIX}${name}`;

    (function injectModalZFix(){
        if (document.getElementById('__modal_zfix__')) return;
        const style = document.createElement('style');
        style.id='__modal_zfix__';
        style.textContent =
            '.modal-backdrop.show:nth-of-type(2){z-index:1055!important}' +
            '.modal.show:nth-of-type(2){z-index:1060!important}';
        document.head.appendChild(style);
    })();

    function getTitleEl(){ return document.getElementById('evtTitle'); }
    function hasSelect2(el){
        try { return !!(window.jQuery && window.jQuery.fn && el && window.jQuery(el).data('select2')); } catch(_){ return false; }
    }
    function getTitleValue(){
        const el = getTitleEl();
        if (!el) return '';
        if (hasSelect2(el)) return String(window.jQuery(el).val() || '').trim();
        return String(el.value || '').trim();
    }
    function setTitleValue(text){
        const el = getTitleEl(); if (!el) return;
        const t = String(text || '').trim();
        const ensureNativeOption = (sel, val) => {
            if (!val) { sel.value = ''; return; }
            let hit = false;
            for (const o of sel.options) {
                if (o.value === val) { o.selected = true; hit = true; }
                else { o.selected = false; }
            }
            if (!hit) sel.add(new Option(val, val, true, true));
        };

        if (hasSelect2(el)) {
            const $el = window.jQuery(el);
            const esc = (window.jQuery.escapeSelector || (s => s.replace(/([;&,.+*~':"!^#$%@\[\]\(\)=>|\/\\])/g,'\\$1')));
            if (t && !$el.find('option[value="'+esc(t)+'"]').length) {
                $el.append(new Option(t, t, true, true));
            }
            $el.val(t || null).trigger('change.select2');
        } else {
            ensureNativeOption(el, t);
        }
    }

    function ensureParticipantModal() {
        if (document.getElementById('participantModal')) return;
        const html = `
<div class="modal fade" id="participantModal" tabindex="-1" role="dialog" aria-hidden="true">
  <div class="modal-dialog modal-lg" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Create participant</h5>
        <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span>×</span></button>
      </div>
      <div class="modal-body">
        <div class="text-danger small" id="pmError" style="display:none;"></div>
        <form id="pmForm" novalidate>
          <div class="form-row">
            <div class="form-group col-md-6">
              <label>First name</label>
              <input type="text" class="form-control" id="pmFirst" required>
            </div>
            <div class="form-group col-md-6">
              <label>Last name</label>
              <input type="text" class="form-control" id="pmLast">
            </div>
          </div>
          <div class="form-group">
            <label>Clinicians</label>
            <select id="pmClinicians" class="form-control select2" multiple data-placeholder="Assign clinicians…"></select>
          </div>
          <div class="form-group">
            <label>Contacts</label>
            <div id="pmContactsWrap"></div>
            <button type="button" class="btn btn-outline-primary btn-sm js-pm-add-contact">Add contact</button>
          </div>
          <div class="form-group">
            <label>Addresses</label>
            <div id="pmAddrList"></div>
            <button type="button" class="btn btn-outline-primary btn-sm" id="pmAddAddress">Add address</button>
          </div>
          <div class="form-row">
            <div class="form-group col-md-6">
              <label>Allergies</label>
              <input type="text" class="form-control" id="pmAllergies">
            </div>
            <div class="form-group col-md-6">
              <label>Medicare</label>
              <input type="text" class="form-control" id="pmMedicare">
            </div>
          </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" id="btnCreateParticipant" class="btn btn-primary">Create</button>
        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
      </div>
    </div>
  </div>
</div>`;
        document.body.insertAdjacentHTML('beforeend', html);

        (function () {
            const PM_FIELDS = {
                '#pmFirst':   { label: 'First name', max: 40, help: 'First name (max 40 chars).' },
                '#pmLast':    { label: 'Last name',  max: 40, help: 'Last name (max 40 chars).' },
                '#pmAllergies': { label: 'Allergies', max: 100, help: 'Allergies (max 100 chars).' },
                '#pmMedicare':  { label: 'Medicare',  max: 20,  help: 'Medicare (digits only, max 20).' , digitsOnly: true }
            };

            const LIMITS = {
                '__phone__': 20,
                '__email__': 100,
                'participant_addresses.*.address_line1': 100,
                'participant_addresses.*.address_line2': 100,
                'participant_addresses.*.city': 50,
                'participant_addresses.*.postcode': 10,
                'participant_addresses.*.country': 60
            };

            function ensureHelp(el, text){
                let host = el.parentElement || el;
                if (host && !host.classList.contains('phone-input') && el.closest('.phone-input')) {
                    host = el.closest('.phone-input');
                }
                let hint = host.querySelector('.help-hint');
                if (!hint){
                    hint = document.createElement('div');
                    hint.className = 'text-muted small help-hint mt-1';
                    host.appendChild(hint);
                }
                hint.textContent = text;
            }

            function ensureErr(el){
                let m = (el.parentElement||el).querySelector('.invalid-feedback.__lenmsg');
                if (!m){
                    m = document.createElement('div');
                    m.className = 'invalid-feedback __lenmsg';
                    m.style.display = 'none';
                    (el.parentElement||el).appendChild(m);
                }
                return m;
            }

            function bindMax(el, max, label){
                if (!el) return;
                if (max){
                    const cur = parseInt(el.getAttribute('maxlength')||'0',10);
                    const next = Number.isFinite(cur) && cur>0 ? Math.min(cur, max) : max;
                    if (!cur || cur!==next) el.setAttribute('maxlength', String(next));
                }
                if (el.dataset.__pmBound!=='1'){
                    ['input','change','blur','keyup'].forEach(ev=>{
                        el.addEventListener(ev, ()=>{
                            const m = parseInt(el.getAttribute('maxlength')||'0',10);
                            if (!m) return;
                            const len = (el.value||'').length;
                            const box = ensureErr(el);
                            if (len>m){
                                el.classList.add('is-invalid');
                                box.style.display = '';
                                box.textContent = `${label}: over limit (${len}/${m}).`;
                            } else if (document.activeElement===el && len===m){
                                el.classList.add('is-invalid');
                                box.style.display = '';
                                box.textContent = `${label}: reached max (${m}).`;
                            } else {
                                el.classList.remove('is-invalid');
                                box.style.display = 'none';
                                box.textContent = '';
                            }
                        });
                    });
                    el.dataset.__pmBound='1';
                }
                el.dispatchEvent(new Event('input', {bubbles:true}));
            }

            function enforceDigits(el){
                el.setAttribute('inputmode','numeric');
                el.setAttribute('pattern','[0-9 ]*');
                if (el.dataset.__pmDigits!=='1'){
                    el.addEventListener('input', ()=>{ el.value = (el.value||'').replace(/[^\d ]+/g,''); });
                    el.dataset.__pmDigits='1';
                }
            }

            function applyIn(root){
                Object.keys(PM_FIELDS).forEach(sel=>{
                    const cfg = PM_FIELDS[sel];
                    root.querySelectorAll(sel).forEach(el=>{
                        bindMax(el, cfg.max, cfg.label);
                        ensureHelp(el, cfg.help);
                        if (cfg.digitsOnly) enforceDigits(el);
                    });
                });

                root.querySelectorAll('.contact-card .js-phone').forEach(el=>{
                    bindMax(el, LIMITS.__phone__, 'Phone');
                    ensureHelp(el, 'Phone number (digits only).');
                    enforceDigits(el);
                });
                root.querySelectorAll('.contact-card .js-email').forEach(el=>{
                    bindMax(el, LIMITS.__email__, 'Email');
                    ensureHelp(el, 'Email (max 100 chars).');
                });

                const addrNodes = root.querySelectorAll('input[name^="participant_addresses"], textarea[name^="participant_addresses"]');
                addrNodes.forEach(el=>{
                    const n = el.getAttribute('name')||'';
                    const map = [
                        ['address_line1','Street address', LIMITS['participant_addresses.*.address_line1'], 'Street address (max 100 chars).'],
                        ['address_line2','Address line 2', LIMITS['participant_addresses.*.address_line2'], 'Address line 2 (max 100 chars).'],
                        ['city','City', LIMITS['participant_addresses.*.city'], 'City (max 50 chars).'],
                        ['postcode','Postcode', LIMITS['participant_addresses.*.postcode'], 'Postcode (max 10 chars).'],
                        ['country','Country', LIMITS['participant_addresses.*.country'], 'Country (max 60 chars).']
                    ];
                    for (const [field,label,max,help] of map){
                        const reDot = new RegExp(`^participant_addresses\\.\\d+\\.${field}$`);
                        const reBrk = new RegExp(`^participant_addresses\\[\\d+\\]\\[${field}\\]$`);
                        if (reDot.test(n) || reBrk.test(n)){
                            bindMax(el, max, label);
                            ensureHelp(el, help);
                            if (field==='postcode') enforceDigits(el);
                            break;
                        }
                    }
                });
            }

            window.__applyPmLimits = applyIn;

            function observeDynamic(modal){
                ['#pmContactsWrap','#pmAddrList'].forEach(sel=>{
                    const t = modal.querySelector(sel);
                    if (!t) return;
                    const mo = new MutationObserver(muts=>{
                        muts.forEach(m=> m.addedNodes && m.addedNodes.forEach(node=>{
                            if (node && node.nodeType===1) applyIn(node);
                        }));
                    });
                    mo.observe(t, { childList:true, subtree:true });
                });
            }
            window.__observePmDynamic = observeDynamic;
        })();

        if (window.jQuery) {
            window.jQuery(document).on('shown.bs.modal', '#participantModal', function () {
                var $dlg = window.jQuery(this), $host = $dlg.find('.modal-content');
                $dlg.find('select.select2').each(function(){
                    var $sel = window.jQuery(this); try{$sel.select2('destroy');}catch(_){}
                    $sel.select2({ width:'100%', dropdownParent:$host, placeholder:$sel.data('placeholder')||'', allowClear:true, closeOnSelect:!$sel.prop('multiple') });
                });

                try {
                    if (window.__applyPmLimits) window.__applyPmLimits(this);
                    if (window.__observePmDynamic) window.__observePmDynamic(this);
                } catch(_) {}

                try {
                    this.querySelectorAll('input,textarea').forEach(function(i){
                        i.dispatchEvent(new Event('input', {bubbles:true}));
                    });
                } catch(_) {}
            });
        }
    }

    ensureParticipantModal();

    const esc = (s) =>
        String(s ?? '').replace(/[&<>"']/g, (m) => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[m]));
    const toJson = (r) => { if (!r.ok) throw new Error(r.statusText || 'Network'); return r.json(); };
    const post = (url, body) =>
        fetch(url, {
            method: 'POST',
            credentials: 'same-origin',
            headers: { 'Content-Type': 'application/json', 'X-CSRF-Token': cfg.CSRF || '' },
            body: JSON.stringify(body || {}),
        }).then(toJson);

    const pad = (n) => (n < 10 ? '0' : '') + n;
    const ymd = (d) => `${d.getFullYear()}-${pad(d.getMonth() + 1)}-${pad(d.getDate())}`;
    const hm  = (d) => `${pad(d.getHours())}:${pad(d.getMinutes())}`;
    const addMinutes = (d, mins) => { const t = new Date(d); t.setMinutes(t.getMinutes() + mins); return t; };
    const buildDateTime = (dateStr, hmStr) => `${dateStr}T${hmStr}:00`;

    const norm = (s) => String(s || '').toLowerCase().replace(/\s+/g, ' ').trim();
    const tokenize = (s) => norm(s).split(' ').filter(Boolean);
    const debounce = (fn, wait=200) => { let t; return (...args) => { clearTimeout(t); t = setTimeout(() => fn.apply(null, args), wait); }; };

    const calEl   = document.getElementById('calendar');
    if (!calEl) return;

    const selClin = document.getElementById('clinician');
    const btnToday= document.getElementById('btnToday');

    const modalId = '#createEventModal';
    const fTitle  = getTitleEl();
    const fType   = document.getElementById('evtType');
    const fServiceT = document.getElementById('evtServiceType');
    const fService  = document.getElementById('evtService');
    const fDate   = document.getElementById('evtDate');
    const fStart  = document.getElementById('evtStart');
    const fEnd    = document.getElementById('evtEnd');
    const fClin   = document.getElementById('evtClinician');
    const fStatus = document.getElementById('evtStatus');
    const fLoc    = document.getElementById('evtLocation');
    const fContact= document.getElementById('evtContact');
    const fRem    = document.getElementById('evtReminder');
    const fRemTpl = document.getElementById('evtReminderTpl');
    const modalTitleEl = document.querySelector('#createEventModal .modal-title');

    const btnSave = document.getElementById('btnUpdateEvent');
    const btnDel  = document.getElementById('btnDeleteEvent');
    const errBox  = document.getElementById('evtError');
    const datalist= document.getElementById('participantsDatalist');

    const step1   = document.getElementById('evtStep1');
    const step2   = document.getElementById('evtStep2');
    const btnNext = document.getElementById('btnNext');
    const btnBack = document.getElementById('btnBack');

    const noteTpl = document.getElementById('noteTpl');
    const noteBody= document.getElementById('noteBody');

    const tabBtnDetails = document.getElementById('btnTabDetails');
    const tabBtnNote    = document.getElementById('btnTabNote');

    const wlModal = '#waitListModal';
    const lnkSelectFromWL = document.getElementById('lnkSelectFromWL');
    const lnkAddToWL      = document.getElementById('lnkAddToWL');
    const wlSearch = document.getElementById('wlSearch');
    const wlClinEl = document.getElementById('wlClin');
    const wlBody   = document.getElementById('wlTableBody');
    const btnWLReload = document.getElementById('btnWLReload');
    const WL_EMPTY_MSG = 'No matching waitlist entries.';

    const pmModal = '#participantModal';
    const pmForm  = document.getElementById('pmForm');
    const pmFirst = document.getElementById('pmFirst');
    const pmLast  = document.getElementById('pmLast');
    const pmAllergies = document.getElementById('pmAllergies');
    const pmMedicare  = document.getElementById('pmMedicare');
    const pmClinSel   = document.getElementById('pmClinicians');
    const pmError     = document.getElementById('pmError');
    const btnCreateParticipant = document.getElementById('btnCreateParticipant');
    const lnkCreateParticipant  = document.getElementById('lnkCreateParticipant');

    const pmContactsWrap = document.getElementById('pmContactsWrap');
    const pmAddContactBtn = document.querySelector('.js-pm-add-contact');

    let collectPMAddresses = () => [];

    const NOTE_TPLS = {
        generic: ['Subjective:', '', 'Objective:', '', 'Assessment:', '', 'Plan:', ''].join('\n'),
        brief:   ['Note:', ''].join('\n'),
    };

    const SERVICE_LABELS = { };
    const formatServiceLabel = (key) => {
        if (!key) return '';
        if (SERVICE_LABELS[key]) return SERVICE_LABELS[key];
        return String(key).replace(/[_-]+/g, ' ').replace(/\b(\w)/g, (m)=>m.toUpperCase());
    };

    let mode = 'create';
    let editingId = null;
    let chosenWaitId = null;

    let pendingLocation = '';
    let __fp;

    let currentParticipantId = null;
    let participantLocOptions = [];

    let lastSuggestList = [];
    let lastResolvedKey = '';
    let pendingQuickAddName = '';
    let orgLocOptions = [];

    const BASE_LOC_GROUP = [
        { label: 'OTHER', items: [
                { id:"Organisation's location", text:"Organisation's location" },
                { id:'Online Consultation', text:'Online Consultation' },
                { id:'Phone Consultation',  text:'Phone Consultation'  },
                { id:'__other__',           text:'Other address'       }
            ]}
    ];

    function showModal(){ $ ? $(modalId).modal('show') : document.querySelector(modalId)?.classList.add('show'); }
    function hideModal(){ $ ? $(modalId).modal('hide') : document.querySelector(modalId)?.classList.remove('show'); }
    function showPM()   { $ ? $(pmModal).modal('show')  : document.querySelector(pmModal)?.classList.add('show'); }
    function hidePM()   { $ ? $(pmModal).modal('hide')  : document.querySelector(pmModal)?.classList.remove('show'); }
    function showWL()   { $ ? $(wlModal).modal('show')  : document.querySelector(wlModal)?.classList.add('show'); }
    function hideWL()   { $ ? $(wlModal).modal('hide')  : document.querySelector(wlModal)?.classList.remove('show'); }

    function clearError(){ if (errBox) { errBox.textContent=''; errBox.style.display='none'; } }
    function showError(msg){ if (!errBox) return; errBox.textContent = msg || 'Error'; errBox.style.display='block'; }

    function updateModalTitle(){
        if (!modalTitleEl) return;
        modalTitleEl.textContent = (mode === 'edit') ? 'Edit Appointment' : 'New Appointment';
    }

    function showStep(n){
        if (step1) step1.style.display = n===1 ? '' : 'none';
        if (step2) step2.style.display = n===2 ? '' : 'none';
        if (btnBack) btnBack.style.display = n===2 ? 'inline-block' : 'none';
        if (btnNext) btnNext.style.display = n===1 ? 'inline-block' : 'none';
        const isDetails = n===1;
        if (tabBtnDetails && tabBtnNote){
            tabBtnDetails.classList.toggle('active', isDetails);
            tabBtnNote.classList.toggle('active', !isDetails);
            tabBtnDetails.setAttribute('aria-pressed', isDetails ? 'true' : 'false');
            tabBtnNote.setAttribute('aria-pressed', !isDetails ? 'true' : 'false');
        }
    }

    (function markClinicianRequired(){
        if (!fClin) return;
        try { fClin.setAttribute('required','required'); } catch(_){}

        const grp = fClin.closest('.form-group') || fClin.parentElement;
        const label = grp ? grp.querySelector('label') : null;
        if (label && !label.querySelector('.req-dot')) {
            const star = document.createElement('span');
            star.className = 'req-dot';
            star.style.cssText = 'color:#d91f1f;margin-left:6px;';
            star.textContent = '*';
            label.appendChild(star);
        }
    })();

    function fullFromParts(x){
        return [x.first_name, x.middle_name, x.last_name]
            .filter(Boolean).join(' ').replace(/\s+/g,' ').trim();
    }
    function displayName(x){
        const byParts = fullFromParts(x);
        return (byParts || x.full_name || x.name || x.preferred || x.email || '').trim();
    }
    function candidateKeys(x){
        const set = new Set();
        [x.name, x.full_name, fullFromParts(x), x.preferred, x.email]
            .forEach(v => { const k = norm(v); if (k) set.add(k); });
        return set;
    }
    function firstLastEqual(typed, candidate){
        const t = tokenize(typed), c = tokenize(candidate);
        if (t.length < 2 || c.length < 2) return false;
        return t[0] === c[0] && t[t.length-1] === c[c.length-1];
    }

    function fetchSuggest(q){
        if (!cfg.PARTS) return Promise.resolve();

        const terms = Array.from(new Set((q || '').split(/\s+/).filter(Boolean)));
        const queries = terms.length > 1 ? terms : [q];

        const req = (term) => {
            const url = new URL(cfg.PARTS, window.location.origin);
            if (term) url.searchParams.set('q', term);
            return fetch(url, { credentials:'same-origin' }).then(toJson).catch(()=>[]);
        };

        return Promise.all(queries.map(req)).then(pages => {
            const merged = [];
            const seen = new Set();
            pages.flat().forEach(x => {
                const idKey = String(x.id ?? (x.email || x.name || Math.random()));
                if (seen.has(idKey)) return;
                seen.add(idKey);
                merged.push(x);
            });
            lastSuggestList = merged;

            if (datalist) {
                const typed = getTitleValue();
                const optsHtml = merged.map(x => {
                    const label = displayName(x);
                    return label ? `<option value="${esc(label)}"></option>` : '';
                }).join('');

                let showAdd = false;
                if (typed) {
                    const key = norm(typed);
                    showAdd = !merged.some(x => {
                        const keys = candidateKeys(x);
                        return keys.has(key) || firstLastEqual(typed, displayName(x));
                    });
                }

                const addHtml = (typed && showAdd)
                    ? `<option value="${esc(buildAddOptValue(typed))}"></option>`
                    : '';

                datalist.innerHTML = optsHtml + addHtml;
            }
            updateQuickAddUI();
        });
    }


    const quickAdd = document.getElementById('titleQuickAdd');
    const quickAddLink = document.getElementById('lnkQuickAdd');

    function updateQuickAddUI(){
        if (quickAdd) quickAdd.style.display = 'none';
        pendingQuickAddName = '';

        if (!fTitle || !datalist) return;

        const raw = getTitleValue();
        if (!raw) return;

        let exists = false;
        if (currentParticipantId) exists = true;
        else {
            for (const x of (lastSuggestList || [])) {
                const keys = candidateKeys(x);
                if (keys.has(norm(raw)) || firstLastEqual(raw, displayName(x))) { exists = true; break; }
            }
        }

        const addVal = buildAddOptValue(raw);
        const hasAdd = !!datalist.querySelector(`option[value="${CSS.escape(addVal)}"]`);

        if (!exists && raw && !hasAdd) {
            datalist.insertAdjacentHTML('beforeend', `<option value="${esc(addVal)}"></option>`);
            pendingQuickAddName = raw;
        } else if ((exists || !raw) && hasAdd) {
            [...datalist.querySelectorAll(`option`)].forEach(op => {
                if (op.value.startsWith(ADD_OPT_PREFIX)) op.remove();
            });
        }
    }

    function openQuickAddModal(name){
        ensureParticipantModal();
        const parts = (name || '').trim().split(/\s+/).filter(Boolean);
        const first = parts.shift() || '';
        const last  = parts.join(' ');

        const _pmFirst = document.getElementById('pmFirst');
        const _pmLast  = document.getElementById('pmLast');
        if (_pmFirst) _pmFirst.value = first;
        if (_pmLast)  _pmLast.value  = last;

        showPM();
        setTimeout(()=>{ try{ document.querySelector('#participantModal input[type=email]')?.focus(); }catch(_){ } }, 0);
    }

    if (quickAddLink){
        quickAddLink.addEventListener('click', (e)=>{
            e.preventDefault();
            if (!pendingQuickAddName) return;
            openQuickAddModal(pendingQuickAddName);
        });
    }
    if (lnkCreateParticipant){
        lnkCreateParticipant.addEventListener('click', (e)=>{
            e.preventDefault();
            openQuickAddModal(getTitleValue());
        });
    }

    async function resolveParticipantIdByTitle(){
        const raw = getTitleValue();
        const key = norm(raw);

        if (key === lastResolvedKey && participantLocOptions.length) {
            updateQuickAddUI();
            try { window.__contacts?.loadForParticipant(currentParticipantId || null); } catch(_){}
            return currentParticipantId || null;
        }

        if (!key){
            currentParticipantId = null;
            participantLocOptions = [];
            rebuildLocationOptions();
            lastResolvedKey = '';
            updateQuickAddUI();
            try { window.__contacts?.loadForParticipant(null); } catch(_){}
            return null;
        }

        const localPick = (() => {
            for (const x of (lastSuggestList || [])) {
                const keys = candidateKeys(x);
                if (keys.has(key) || firstLastEqual(raw, displayName(x))) return x;
            }
            return null;
        })();

        if (localPick && localPick.id){
            currentParticipantId = Number(localPick.id);
            lastResolvedKey = key;
            await loadParticipantLocations(currentParticipantId);
            updateQuickAddUI();
            try { window.__contacts?.loadForParticipant(currentParticipantId); } catch(_){}
            return currentParticipantId;
        }

        try{
            const words = tokenize(raw);
            const queries = words.length > 1 ? words : [raw];
            const req = (term) => {
                const u = new URL(cfg.PARTS, window.location.origin);
                u.searchParams.set('q', term);
                return fetch(u, { credentials:'same-origin' }).then(toJson).catch(()=>[]);
            };
            const merged = (await Promise.all(queries.map(req))).flat();

            let pick = null;
            for (const x of merged){
                const keys = candidateKeys(x);
                if (keys.has(key) || firstLastEqual(raw, displayName(x))) { pick = x; break; }
            }

            if (pick && pick.id){
                currentParticipantId = Number(pick.id);
                lastResolvedKey = key;
                await loadParticipantLocations(currentParticipantId);
                updateQuickAddUI();
                try { window.__contacts?.loadForParticipant(currentParticipantId); } catch(_){}
                return currentParticipantId;
            }
        }catch(e){
            console.warn('resolveParticipantIdByTitle fallback failed:', e);
        }

        currentParticipantId = null;
        participantLocOptions = [];
        rebuildLocationOptions();
        lastResolvedKey = key;
        updateQuickAddUI();
        try { window.__contacts?.loadForParticipant(null); } catch(_){}
        return null;
    }

    async function loadParticipantLocations(pid){
        participantLocOptions = [];
        if (!pid || !cfg.PART_ADDR) { rebuildLocationOptions(); return; }

        try {
            const url = new URL(cfg.PART_ADDR, window.location.origin);
            url.searchParams.set('participant_id', String(pid));
            const rows = await fetch(url, { credentials: 'same-origin' }).then(r => r.json());

            participantLocOptions = Array.isArray(rows)
                ? rows.map(x => {
                    if (typeof x === 'string') {
                        const v = String(x);
                        return { value: v, text: v };
                    }
                    if (x && typeof x === 'object') {
                        const text =
                            x.text ||
                            x.label ||
                            x.address ||
                            [x.addr1 || x.address1, x.addr2 || x.address2, x.city, x.region || x.state, x.postcode, x.country]
                                .filter(Boolean)
                                .join(', ');
                        if (!text) return null;
                        const v = String(x.value || text);
                        return { value: v, text };
                    }
                    return null;
                }).filter(Boolean)
                : [];
        } catch (e) {
            participantLocOptions = [];
        }

        rebuildLocationOptions();
    }

    async function reloadWL(){
        if (!cfg.WL_FEED) return;
        const url = new URL(cfg.WL_FEED, window.location.origin);
        const q = (wlSearch && wlSearch.value.trim()) || '';
        if (q) url.searchParams.set('q', q);
        if (wlClinEl && wlClinEl.value) url.searchParams.set('clinician_id', wlClinEl.value);
        try{
            const list = await fetch(url, { credentials:'same-origin' }).then(toJson);
            if (!list || list.length===0){
                wlBody.innerHTML = `<tr><td colspan="6" class="text-center text-muted py-3">${WL_EMPTY_MSG}</td></tr>`;
                return;
            }
            wlBody.innerHTML = list.map((r,i)=>`
        <tr>
          <td>${i+1}</td>
          <td>${esc(r.name||'')}</td>
          <td>${esc(r.clinician||'')}</td>
          <td>${esc(r.desired_date||'')}</td>
          <td class="text-truncate" style="max-width:220px;">${esc(r.notes||'')}</td>
          <td class="text-right"><button type="button" class="btn btn-sm btn-primary" data-use="${r.id}">Use</button></td>
        </tr>`).join('');
        }catch(e){
            wlBody.innerHTML = `<tr><td colspan="6" class="text-center text-danger py-3">Load failed</td></tr>`;
            console.warn('WL load failed', e);
        }
    }

    if (wlBody){
        wlBody.addEventListener('click', async (e)=>{
            const btn = e.target.closest('[data-use]'); if (!btn) return;
            const id  = btn.getAttribute('data-use');
            const tr  = btn.closest('tr');
            const name = tr.children[1]?.textContent.trim() || '';
            const clinTxt = tr.children[2]?.textContent.trim() || '';
            setTitleValue(name);
            await resolveParticipantIdByTitle();
            try { window.__contacts?.loadForParticipant(currentParticipantId || null); } catch(_){}
            if (fClin && clinTxt){
                for (const opt of fClin.options){ if (opt.textContent.trim()===clinTxt){ fClin.value = opt.value; break; } }
            }
            chosenWaitId = Number(id);
            hideWL();
            setTimeout(()=>{ try{ getTitleEl()?.focus(); }catch(_){ } }, 0);
        });
    }

    if (lnkAddToWL) {
        lnkAddToWL.addEventListener('click', async (e) => {
            e.preventDefault();
            clearError && clearError();

            const name       = (getTitleValue && getTitleValue()) || '';
            const clinician  = (fClin && fClin.value) ? Number(fClin.value) : null;
            const desired    = (fDate && fDate.value) || null;
            const contact    =
                (typeof window.__getSelectedContact === 'function'
                    ? window.__getSelectedContact()
                    : (document.getElementById('evtContact')?.value || '')).trim();
            const notes      = (noteBody && noteBody.value || '').trim();

            // 这里强制要求 Clinician（你之前也提过 Clinician 要必填）
            if (!clinician) {
                showError && showError('Please choose a clinician before adding to wait list.');
                return;
            }
            if (!name && !contact) {
                showError && showError('Please enter a title (participant name) or a contact.');
                return;
            }

            const payload = {
                name: name || contact,
                clinician_id: clinician,
                desired_date: desired,
                contact: contact || null,
                notes: notes || null,
                status: 'waiting'
            };

            try {
                const res = await fetch(cfg.WL_ADD, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: { 'Content-Type': 'application/json', 'X-CSRF-Token': (cfg.CSRF || '') },
                    body: JSON.stringify(payload)
                });
                if (!res.ok) {
                    const j = await res.json().catch(()=>null);
                    throw new Error(j?.error || 'Add to wait list failed');
                }
                // 成功后给个提示，必要时可自动打开 waitlist 弹窗
                if (window.jQuery) {
                    window.jQuery.growl?.notice?.({ message: 'Added to wait list.' });
                } else {
                    alert('Added to wait list.');
                }
            } catch (err) {
                showError && showError(err.message || 'Add to wait list failed');
            }
        });
    }

    if (btnWLReload) btnWLReload.addEventListener('click', reloadWL);
    if (wlSearch){
        wlSearch.addEventListener('keydown', (e)=>{ if (e.key==='Enter'){ e.preventDefault(); reloadWL(); } });
    }
    if (lnkSelectFromWL){
        lnkSelectFromWL.addEventListener('click', (e)=>{ e.preventDefault(); reloadWL().then(showWL); });
    }

    function rebuildLocationOptions() {
        if (!fLoc) return;

        const keepVal = (window.jQuery && $.fn && $.fn.select2)
            ? String($(fLoc).val() || '')
            : (fLoc.value || '');

        fLoc.innerHTML = '<option></option>';

        const seen = new Set();

        if (participantLocOptions.length) {
            const og = document.createElement('optgroup');
            og.label = 'Participant Addresses';
            participantLocOptions.forEach(it => {
                const v = String(it.value);
                if (seen.has(v)) return;
                seen.add(v);
                const o = document.createElement('option');
                o.value = v;
                o.textContent = it.text || v;
                og.appendChild(o);
            });
            fLoc.appendChild(og);
        }

        if (window.orgLocOptions && window.orgLocOptions.length) {
            const og = document.createElement('optgroup');
            og.label = "Organisation's location";
            window.orgLocOptions.forEach(it => {
                const v = String(it.value);
                if (seen.has(v)) return;
                seen.add(v);
                const o = document.createElement('option');
                o.value = v;
                o.textContent = it.text || v;
                og.appendChild(o);
            });
            fLoc.appendChild(og);
        }

        const other = document.createElement('optgroup');
        other.label = 'OTHER';

        if (!window.orgLocOptions || window.orgLocOptions.length === 0) {
            const oOrg = document.createElement('option');
            oOrg.value = "Organisation's location";
            oOrg.textContent = "Organisation's location";
            if (!seen.has(oOrg.value)) { seen.add(oOrg.value); other.appendChild(oOrg); }
        }

        const fixed = [
            { id: 'Online Consultation', text: 'Online Consultation' },
            { id: 'Phone Consultation',  text: 'Phone Consultation'  },
            { id: '__other__',           text: 'Other address'       }
        ];
        fixed.forEach(it => {
            if (seen.has(it.id)) return;
            seen.add(it.id);
            const o = document.createElement('option');
            o.value = it.id;
            o.textContent = it.text;
            other.appendChild(o);
        });
        fLoc.appendChild(other);

        if (window.jQuery && $.fn && $.fn.select2) {
            const $sel = $(fLoc);
            if (!$sel.data('select2')) {
                $sel.select2({
                    width: '100%',
                    allowClear: true,
                    placeholder: $sel.data('placeholder') || 'Search or pick a location…',
                    dropdownParent: $('#createEventModal')
                });
            }
            if (keepVal && !$sel.find('option[value="' + $.escapeSelector(keepVal) + '"]').length) {
                $sel.append(new Option(keepVal, keepVal, true, true));
            }
            $sel.val(keepVal || null).trigger('change.select2');
        } else {
            fLoc.value = keepVal || '';
        }
    }

    if (fLoc){
        rebuildLocationOptions();
        if ($ && $.fn.select2){
            const $sel = $(fLoc);
            if (!$sel.data('select2')) {
                $sel.select2({
                    width:'100%',
                    allowClear:true,
                    placeholder: $sel.data('placeholder') || 'Search or pick a location…',
                    dropdownParent: $('#createEventModal')
                });
            }
            $sel.on('change', function(){
                if (String($(this).val()) === '__other__'){
                    $(this).val(null).trigger('change');
                    $('#locationModal').modal('show');
                }
            });
        } else {
            fLoc.addEventListener('change', function(){
                if ((this.value||'') === '__other__'){
                    this.value='';
                    document.querySelector('#locationModal')?.classList.add('show');
                }
            });
        }
    }

    const locType    = document.getElementById('locType');
    const locAddr1   = document.getElementById('locAddr1');
    const locAddr2   = document.getElementById('locAddr2');
    const locCity    = document.getElementById('locCity');
    const locRegion  = document.getElementById('locRegion');
    const locPostcode= document.getElementById('locPostcode');
    const locCountry = document.getElementById('locCountry');
    const btnLocSave = document.getElementById('btnLocSave');

    function composeAddress(){
        const t = (locType?.value||'').trim();
        if (t && t !== 'other'){
            return ({ org: "Organisation's location", online: 'Online Consultation', phone: 'Phone Consultation' })[t] || '';
        }
        const a1 = (locAddr1?.value||'').trim();
        const a2 = (locAddr2?.value||'').trim();
        const city= (locCity?.value||'').trim();
        const reg = (locRegion?.value||'').trim();
        const pc  = (locPostcode?.value||'').trim();
        const ctry= (locCountry?.value||'').trim();
        const parts = [];
        if (a1) parts.push(a1);
        if (a2) parts.push(a2);
        const tail = [city, reg, pc].filter(Boolean).join(' ');
        if (tail) parts.push(tail);
        if (ctry) parts.push(ctry);
        return parts.join(', ');
    }

    function setLocationSelectValue(val){
        if (!fLoc) return;
        const v = val == null ? '' : String(val);
        if ($ && $.fn.select2){
            const $sel = $(fLoc);
            if (v && !$sel.find('option[value="'+$.escapeSelector(v)+'"]').length){
                $sel.append(new Option(v, v, true, true));
            }
            $sel.val(v || null).trigger('change');
        } else {
            if (v && ![...fLoc.options].some(o=>o.value===v)){
                const o = document.createElement('option'); o.value = v; o.textContent = v; fLoc.appendChild(o);
            }
            fLoc.value = v || '';
        }
    }
    if (btnLocSave){
        btnLocSave.addEventListener('click', function(){
            const val = composeAddress();
            if (!val){ alert('Please complete the location.'); return; }
            setLocationSelectValue(val);
            $('#locationModal').modal('hide');
        });
    }
    window.__setLocationSelectValue = setLocationSelectValue;

    const PHONE_TAGS = {mobile:'Mobile',home:'Home',work:'Work',personal:'Personal',other:'Other'};
    const EMAIL_TAGS = {personal:'Personal',work:'Work',other:'Other'};

    function optionsHtml(map, sel){ return Object.keys(map).map(k=>`<option value="${k}"${k===sel?' selected':''}>${map[k]}</option>`).join(''); }

    function pmNextStart(){
        const wrap = document.getElementById('pmContactsWrap');
        const cards = wrap ? wrap.querySelectorAll('.pm-contact-card') : [];
        if (!cards || !cards.length) return 0;
        const last = cards[cards.length-1];
        return (parseInt(last.dataset.start,10) || 0) + 2;
    }
    function pmRenumber(){
        const wrap = document.getElementById('pmContactsWrap');
        if (!wrap) return;
        wrap.querySelectorAll('.pm-contact-card').forEach((c,i)=>{
            const t = c.querySelector('strong'); if (t) t.textContent = 'Contact ' + (i+1);
        });
    }
    function pmCardTpl(start){
        return `
    <div class="pm-contact-card border rounded p-2 mb-2" data-start="${start}">
      <div class="d-flex align-items-center mb-2" style="gap:8px;border-bottom:1px dashed #e5e7eb;padding-bottom:8px;">
        <strong class="text-muted small mb-0">Contact</strong>
        <div class="ml-2">
          <select class="form-control form-control-sm js-rel" style="min-width:160px">
            <option value="self">Self</option>
            <option value="parent">Parent</option>
            <option value="spouse">Spouse/Partner</option>
            <option value="sibling">Sibling</option>
            <option value="child">Child</option>
            <option value="relative">Relative</option>
            <option value="friend">Friend</option>
            <option value="guardian">Guardian</option>
            <option value="caregiver">Caregiver</option>
            <option value="other">Other</option>
          </select>
        </div>
        <div class="ml-auto d-flex align-items-center">
          <input class="form-check-input js-card-primary" type="radio" name="pm_contacts[_primary_index]" value="${start}">
          <span class="small text-muted ml-1">Primary</span>
          <button type="button" class="btn btn-outline-danger btn-sm ml-2 js-card-del">Remove</button>
        </div>
      </div>

      <div class="form-row">
        <div class="col-md-4">
          <label class="small text-muted">Phone label</label>
          <select class="form-control form-control-sm js-tag-phone">${optionsHtml(PHONE_TAGS,'mobile')}</select>
        </div>
        <div class="col-md-8">
          <label class="small text-muted">Phone</label>
          <input type="text" class="form-control js-phone" placeholder="+61 xxx xxx xxx">
        </div>
      </div>

      <div class="form-row mt-2">
        <div class="col-md-4">
          <label class="small text-muted">Email label</label>
          <select class="form-control form-control-sm js-tag-email">${optionsHtml(EMAIL_TAGS,'personal')}</select>
        </div>
        <div class="col-md-8">
          <label class="small text-muted">Email</label>
          <input type="email" class="form-control js-email" placeholder="name@example.com">
        </div>
      </div>
    </div>`;
    }

    function collectPMContacts(){
        const wrap = document.getElementById('pmContactsWrap');
        const cards = wrap ? [...wrap.querySelectorAll('.pm-contact-card')] : [];
        const primaryIndex = (document.querySelector('input[name="pm_contacts[_primary_index]"]:checked')?.value || '').trim();
        const out = [];
        cards.forEach(card=>{
            const start = parseInt(card.dataset.start,10) || 0;
            const rel   = (card.querySelector('.js-rel')?.value || 'self').trim();
            const ptag  = (card.querySelector('.js-tag-phone')?.value || 'mobile').trim();
            const etag  = (card.querySelector('.js-tag-email')?.value || 'personal').trim();
            const phone = (card.querySelector('.js-phone')?.value || '').trim();
            const email = (card.querySelector('.js-email')?.value || '').trim();
            const isPrimaryCard = String(start) === primaryIndex;

            if (phone) {
                out.push({ type:'phone', value:phone, label:`${rel}|${ptag}`, sort:start*10, is_primary: isPrimaryCard ? 1 : 0 });
            }
            if (email) {
                out.push({ type:'email', value:email, label:`${rel}|${etag}`, sort:start*10+1, is_primary: isPrimaryCard ? 1 : 0 });
            }
        });
        if (!out.some(x => x.is_primary === 1) && out.length) out[0].is_primary = 1;
        return out;
    }

    if (pmAddContactBtn && pmContactsWrap){
        pmAddContactBtn.addEventListener('click', ()=>{
            const start = pmNextStart();
            pmContactsWrap.insertAdjacentHTML('beforeend', pmCardTpl(start));
            pmRenumber();
        });
        pmContactsWrap.addEventListener('click', (e)=>{
            const del = e.target.closest('.js-card-del');
            if (!del) return;
            del.closest('.pm-contact-card')?.remove();
            pmRenumber();
        });
        pmRenumber();
    }

    const pmAddrList      = document.getElementById('pmAddrList');
    const pmAddAddressBtn = document.getElementById('pmAddAddress');

    function addrNextIndex(){
        const cards = pmAddrList ? pmAddrList.querySelectorAll('.pm-addr-card') : [];
        let max = -1;
        cards.forEach(c => { const i = parseInt(c.dataset.index,10); if (!isNaN(i) && i>max) max = i; });
        return max + 1;
    }
    function addrRenumber(){
        if (!pmAddrList) return;
        const cards = pmAddrList.querySelectorAll('.pm-addr-card');
        cards.forEach((c,i)=>{
            const title = c.querySelector('strong');
            if (title) title.textContent = 'Address ' + (i+1);
            const del = c.querySelector('.js-addr-del');
            if (del) del.style.display = (cards.length > 1 ? '' : 'none');
        });
    }
    function addrCardTpl(index){
        return `
  <div class="pm-addr-card border rounded p-2 mb-2" data-index="${index}">
    <div class="d-flex align-items-center mb-2" style="gap:8px;border-bottom:1px dashed #e5e7eb;padding-bottom:8px;">
      <strong class="text-muted small mb-0">Address</strong>
      <div class="ml-2" style="min-width:160px;">
        <select class="form-control form-control-sm js-addr-type">
          <option value="home" selected>Home</option>
          <option value="work">Work</option>
          <option value="other">Other</option>
        </select>
      </div>
      <button type="button" class="btn btn-outline-danger btn-sm ml-auto js-addr-del">Remove</button>
    </div>
    <div class="form-group mb-2">
      <label class="small text-muted">Street Address</label>
      <input type="text" class="form-control js-addr-l1" placeholder="Address Line 1">
    </div>
    <div class="form-group mb-2">
      <label class="small text-muted">Address Line 2</label>
      <input type="text" class="form-control js-addr-l2" placeholder="Address Line 2">
    </div>
    <div class="form-row">
      <div class="form-group col-md-4">
        <label class="small text-muted">City</label>
        <input type="text" class="form-control js-addr-city" placeholder="City">
      </div>
      <div class="form-group col-md-4">
        <label class="small text-muted">State / Region</label>
        <select class="form-control js-addr-region">
          <option value="">Select State</option>
          <option>ACT</option><option>NSW</option><option>NT</option><option>QLD</option>
          <option>SA</option><option>TAS</option><option>VIC</option><option>WA</option>
        </select>
      </div>
      <div class="form-group col-md-4">
        <label class="small text-muted">Postcode</label>
        <input type="text" class="form-control js-addr-postcode" placeholder="Postcode">
      </div>
    </div>
    <div class="form-row">
      <div class="form-group col-md-6">
        <label class="small text-muted">Country</label>
        <input type="text" class="form-control js-addr-country" value="Australia">
      </div>
      <div class="form-group col-md-6">
        <label class="small text-muted">Time Zone</label>
        <select class="form-control js-addr-tz">
          <option value="">Select a time zone</option>
          <option>Australia/Melbourne</option><option>Australia/Sydney</option><option>Australia/Brisbane</option>
          <option>Australia/Perth</option><option>Australia/Adelaide</option><option>Australia/Hobart</option>
        </select>
      </div>
    </div>
  </div>`;
    }

    if (pmAddAddressBtn && pmAddrList){
        pmAddAddressBtn.addEventListener('click', ()=>{
            pmAddrList.insertAdjacentHTML('beforeend', addrCardTpl(addrNextIndex()));
            addrRenumber();
        });
        pmAddrList.addEventListener('click', (e)=>{
            const btn = e.target.closest('.js-addr-del');
            if (!btn) return;
            btn.closest('.pm-addr-card')?.remove();
            addrRenumber();
        });
        addrRenumber();
    }

    collectPMAddresses = function(){
        const out = [];
        if (!pmAddrList) return out;
        pmAddrList.querySelectorAll('.pm-addr-card').forEach(card=>{
            const type  = (card.querySelector('.js-addr-type')?.value || 'home').trim();
            const l1    = (card.querySelector('.js-addr-l1')?.value || '').trim();
            const l2    = (card.querySelector('.js-addr-l2')?.value || '').trim();
            const city  = (card.querySelector('.js-addr-city')?.value || '').trim();
            const reg   = (card.querySelector('.js-addr-region')?.value || '').trim();
            const pc    = (card.querySelector('.js-addr-postcode')?.value || '').trim();
            const cty   = (card.querySelector('.js-addr-country')?.value || '').trim();
            const tz    = (card.querySelector('.js-addr-tz')?.value || '').trim();

            const allEmpty = !l1 && !l2 && !city && !reg && !pc && !cty && !tz;
            if (allEmpty) return;

            out.push({
                address_type:  ['home','work','other'].includes(type) ? type : 'home',
                address_line1: l1 || null,
                address_line2: l2 || null,
                city:          city || null,
                region:        reg || null,
                postcode:      pc || null,
                country:       cty || 'Australia',
                time_zone:     tz || null,
            });
        });
        return out;
    };

    if (btnCreateParticipant){
        btnCreateParticipant.addEventListener('click', async ()=>{
            const _pmError = document.getElementById('pmError');
            const _pmFirst = document.getElementById('pmFirst');
            const _pmLast  = document.getElementById('pmLast');
            const _pmAllergies = document.getElementById('pmAllergies');
            const _pmMedicare  = document.getElementById('pmMedicare');
            const _pmClinSel   = document.getElementById('pmClinicians');

            if (_pmError){ _pmError.style.display='none'; _pmError.textContent=''; }

            const firstName = (_pmFirst?.value || '').trim();
            if (!firstName){
                _pmError.textContent = 'First name is required.';
                _pmError.style.display = 'block';
                return;
            }

            const contacts  = collectPMContacts();
            if (!contacts.length){
                _pmError.textContent = 'Please enter at least one contact (phone or email).';
                _pmError.style.display = 'block';
                return;
            }
            const firstEmail = (contacts.find(c => c.type === 'email')?.value || '').trim();
            if (!firstEmail){
                _pmError.textContent = 'Email is required. Please enter an email in any contact card.';
                _pmError.style.display = 'block';
                return;
            }

            const pmAddresses = (typeof collectPMAddresses === 'function') ? collectPMAddresses() : [];
            const clinicianIds = Array.from(_pmClinSel?.selectedOptions || [])
                .map(o => Number(o.value)).filter(Boolean);

            const payload = {
                first_name: firstName,
                last_name:  (_pmLast?.value || '').trim(),
                allergies:  (_pmAllergies?.value || '').trim(),
                medicare:   (_pmMedicare?.value || '').trim(),
                contacts,
                participant_addresses: pmAddresses,
                clinician_ids: clinicianIds
            };

            try {
                const res = await fetch(cfg.PART_CREATE, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: { 'Content-Type':'application/json', 'X-CSRF-Token': (cfg.CSRF || '') },
                    body: JSON.stringify(payload)
                });

                if (res.status === 422) {
                    const j = await res.json().catch(()=>null);
                    const errs = (j && j.errors) || {};
                    const lines = [];
                    (function walk(node, path){
                        Object.keys(node||{}).forEach(k=>{
                            const p = path ? (/\D/.test(k) ? `${path}.${k}` : `${path}[${k}]`) : k;
                            const v = node[k];
                            if (v && typeof v === 'object') walk(v, p);
                            else lines.push(`${p}: ${String(v)}`);
                        });
                    })(errs,'');
                    throw new Error(lines.join('\n') || 'Validation failed');
                }

                const j = await res.json().catch(()=>null);
                if (!res.ok || !j) throw new Error((j && (j.error || j.message)) || 'Create failed');

                const pid = Number(j.id ?? j.participant?.id ?? j.data?.id ?? 0);
                const fullname =
                    (j.name ?? j.participant?.name ?? j.full_name) ||
                    `${payload.first_name} ${payload.last_name}`.trim();

                if (fullname) setTitleValue(fullname);
                if (datalist && fullname) {
                    datalist.insertAdjacentHTML('beforeend', `<option value="${fullname.replace(/"/g,'&quot;')}"></option>`);
                }

                hidePM();
                showStep(1);
                currentParticipantId = pid;

                if (pid) {
                    await loadParticipantLocations(pid);
                    await window.__contacts.loadForParticipant(pid);
                }

                const primaryValue =
                    (contacts.find(c => c.is_primary === 1)?.value) ||
                    (contacts.find(c => c.type === 'email')?.value) ||
                    (contacts[0]?.value);

                if (primaryValue) {
                    if (window.jQuery && window.jQuery.fn && window.jQuery.fn.select2) {
                        const $sel = window.jQuery('#evtContact');
                        if (!$sel.find('option[value="'+primaryValue.replace(/"/g,'\\"')+'"]').length) {
                            $sel.append(new Option(primaryValue, primaryValue, true, true));
                        }
                        $sel.val(primaryValue).trigger('change');
                    } else {
                        const sel = document.getElementById('evtContact');
                        if (sel) {
                            let exists = Array.from(sel.options).some(o => o.value === primaryValue);
                            if (!exists) sel.add(new Option(primaryValue, primaryValue, true, true));
                            sel.value = primaryValue;
                        }
                    }
                }

            } catch (err) {
                const _pmError2 = document.getElementById('pmError');
                _pmError2.textContent = (err && err.message) || 'Create failed';
                _pmError2.style.display = 'block';
            }
        });
    }
    ;(()=> {
        const $ = window.jQuery;
        const fDate  = document.getElementById('evtDate');
        const fStart = document.getElementById('evtStart');
        const fEnd   = document.getElementById('evtEnd');

        function insertBillingUI(){
            if (document.getElementById('evtBillingMode')) return;
            const anchor = (fDate && (fDate.closest('.form-group') || fDate.parentElement));
            if (!anchor || !anchor.parentNode) return;

            const wrap = document.createElement('div');
            wrap.className = 'form-group';
            wrap.innerHTML = `
      <label class="d-block mb-1">Billing Mode</label>
      <div id="evtBillingMode" class="d-flex align-items-center" style="gap:16px">
        <label class="mb-0" style="cursor:pointer;">
          <input type="radio" name="billing_mode" value="prorata" checked> Pro-rata
        </label>
        <label class="mb-0" style="cursor:pointer;">
          <input type="radio" name="billing_mode" value="fixed"> Fixed
        </label>
      </div>
      <small class="form-text text-muted">Fixed:Only set the start time; Pro-rata: Automatically add the end time based on service duration.</small>
    `;
            anchor.parentNode.insertBefore(wrap, anchor);
        }

        function findEndContainer(){
            if (!fEnd) return null;
            return fEnd.closest('.col') || fEnd.closest('.form-group') || fEnd.parentElement;
        }

        function pad(n){ return n<10 ? ('0'+n) : String(n); }
        function addMinutesToHM(hm, mins){
            const [h,m] = (hm || '00:00').split(':').map(v=>parseInt(v,10)||0);
            const d = new Date(2000,0,1,h,m,0,0);
            d.setMinutes(d.getMinutes() + (parseInt(mins,10)||0));
            return `${pad(d.getHours())}:${pad(d.getMinutes())}`;
        }

        function selectedServiceDuration(){
            const sel = document.getElementById('evtService');
            const opt = sel && sel.selectedOptions && sel.selectedOptions[0];
            return parseInt(opt?.dataset?.duration || '0', 10) || 0;
        }

        function isFixed(){
            const r = document.querySelector('#evtBillingMode input[value="fixed"]');
            return !!(r && r.checked);
        }

        function setMode(mode){
            const fixed = (mode === 'fixed');
            const rFixed  = document.querySelector('#evtBillingMode input[value="fixed"]');
            const rProrata= document.querySelector('#evtBillingMode input[value="prorata"]');
            if (rFixed && rProrata){ rFixed.checked = fixed; rProrata.checked = !fixed; }

            const endWrap = findEndContainer();
            if (fixed){
                if (endWrap) endWrap.style.display = 'none';
                if (fEnd){ fEnd.value=''; fEnd.setAttribute('data-disabled-by-billing','1'); }
            } else {
                if (endWrap) endWrap.style.display = '';
                if (fEnd) fEnd.removeAttribute('data-disabled-by-billing');
                // 回到 Pro-rata 时，如无 End 则按服务时长补
                const dur = selectedServiceDuration();
                if (dur > 0 && fDate?.value && fStart?.value && fEnd && !fEnd.value){
                    fEnd.value = addMinutesToHM(fStart.value, dur);
                }
            }
        }

        function onModeChange(){
            setMode(isFixed() ? 'fixed' : 'prorata');
        }

        function onStartChange(){
            if (isFixed()) return;
            if (!fEnd || fEnd.value) return;
            const dur = selectedServiceDuration();
            if (dur > 0 && fStart?.value) fEnd.value = addMinutesToHM(fStart.value, dur);
        }

        insertBillingUI();
        setMode('prorata');

        const modeWrap = document.getElementById('evtBillingMode');
        if (modeWrap) modeWrap.addEventListener('change', onModeChange);
        if (fStart){
            fStart.addEventListener('change', onStartChange);
            fStart.addEventListener('input',  onStartChange);
        }

        if ($){
            $(document).on('shown.bs.modal', '#createEventModal', function(){
                const hasEnd = !!(fEnd && (fEnd.value||'').trim());
                setMode(hasEnd ? 'prorata' : 'fixed');
            });
        }

        const btnSave = document.getElementById('btnUpdateEvent');
        if (btnSave){
            btnSave.addEventListener('click', function(){
                if (isFixed() && fEnd){ fEnd.value = ''; }
            });
        }

        window.__billingIsFixed = isFixed;
    })();

    const calendar = new FullCalendar.Calendar(calEl, {
        initialView: 'timeGridWeek',
        locale: 'en',
        height: 'auto',
        expandRows: true,
        slotMinTime: '09:00:00',
        slotMaxTime: '21:30:00',
        slotDuration: '00:15:00',
        scrollTime: '08:00:00',
        slotLabelInterval: '00:30:00',
        scrollTimeReset: false,
        customButtons: {
            jumpDate: {
                text: 'Pick date',
                click: function () { try{ if (window.flatpickr){ openFlatpickrPicker(); } else { openNativePicker(); } }catch(_){ openNativePicker(); } }
            }
        },
        headerToolbar: { left:'prev,next today jumpDate', center:'title', right:'timeGridDay,timeGridWeek,dayGridMonth,listMonth' },

        editable: true,
        selectable: true,
        selectMirror: true,
        unselectAuto: true,

        eventTimeFormat: { hour:'numeric', minute:'2-digit', meridiem:'short' },

        events: (info, success, failure) => {
            try {
                const u = new URL(cfg.FEED, window.location.origin);
                const s = info.start.toISOString().slice(0, 19);
                const e = info.end.toISOString().slice(0, 19);
                u.searchParams.set('start', s);
                u.searchParams.set('end',   e);
                if (selClin && selClin.value) u.searchParams.set('clinician_id', selClin.value);
                fetch(u.toString(), { credentials: 'same-origin' })
                    .then(toJson)
                    .then(success)
                    .catch(failure);
            } catch (err) {
                failure(err);
            }
        },

        dateClick: (info) => {
            mode = 'create'; editingId = null; chosenWaitId = null; window.__calMode = 'create'; clearError();
            updateModalTitle();
            showModal();

            const d = info.date;
            if (fDate)  fDate.value  = ymd(d);
            if (info.view.type.startsWith('timeGrid')){ if (fStart) fStart.value = hm(d); if (fEnd) fEnd.value = hm(addMinutes(d, 30)); }
            else { if (fStart) fStart.value = '09:00'; if (fEnd) fEnd.value = '09:30'; }

            setTitleValue('');
            if (fStatus) fStatus.value = 'pending';
            if (fType)    fType.value = 'standard';
            if (fService) {
                if (fService.tagName === 'SELECT') fService.value = '';
                else fService.value = '';
                if ($ && $.fn.select2 && $(fService).data('select2')) $(fService).val(null).trigger('change');
            }
            if (fClin)    fClin.value = selClin && selClin.value ? selClin.value : '';
            pendingLocation = ''; setLocationSelectValue('');
            if (noteBody) noteBody.value = '';

            if (btnDel)  btnDel.style.display  = 'none';
            if (btnSave) btnSave.textContent   = 'Create';

            fetchSuggest('');
            updateQuickAddUI();
            showStep(1);
            showModal();
        },

        select: ({start, end, allDay, view}) => {
            mode = 'create'; editingId = null; chosenWaitId = null; window.__calMode = 'create'; clearError();
            updateModalTitle();
            showModal();
            if (fDate) fDate.value = ymd(start);
            if (allDay || !view.type.startsWith('timeGrid')){ if (fStart) fStart.value = '09:00'; if (fEnd) fEnd.value = '09:30'; }
            else { if (fStart) fStart.value = hm(start); if (fEnd) fEnd.value = hm(end); }

            setTitleValue('');
            if (fStatus) fStatus.value = 'pending';
            if (fType)    fType.value = 'standard';
            if (fService) {
                if (fService.tagName === 'SELECT') fService.value = '';
                else fService.value = '';
                if ($ && $.fn.select2 && $(fService).data('select2')) $(fService).val(null).trigger('change');
            }
            if (fClin)    fClin.value = selClin && selClin.value ? selClin.value : '';
            pendingLocation = ''; setLocationSelectValue('');
            if (noteBody) noteBody.value = '';

            if (btnDel)  btnDel.style.display  = 'none';
            if (btnSave) btnSave.textContent   = 'Create';

            fetchSuggest('');
            updateQuickAddUI();
            showStep(1);
            showModal();
        },

        eventClick: ({ event }) => {
            mode = 'edit';
            updateModalTitle();
            showModal();
            editingId = event.id;
            chosenWaitId = null;
            window.__calMode = 'edit';
            clearError();

            const xp = event.extendedProps || {};

            setTitleValue((event.title || xp.participant_name || '').trim());
            if (fDate)   fDate.value   = ymd(event.start);
            if (fStart)  fStart.value  = hm(event.start);
            if (fEnd)    fEnd.value    = event.end ? hm(event.end) : '';
            if (fStatus) fStatus.value = xp.status || 'pending';
            if (fType)   fType.value   = xp.appointment_type || 'standard';
            if (fClin)   fClin.value   = xp.user_id || '';

            const wantSvcId   = xp.service_id != null && xp.service_id !== '' ? String(xp.service_id) : null;
            const wantSvcText = String(
                xp.service_name || xp.service_type || xp.service || ''
            ).replace(/^name:/, '');
            function selectServiceInDom(id, text) {
                const el = fService || document.getElementById('evtService');
                if (!el) return;
                const cleanText = (text || '').replace(/^name:/,'').trim();

                if (window.jQuery && window.jQuery.fn) {
                    const $sel = window.jQuery(el);
                    const setVal = (val, label) => {
                        if (val && !$sel.find('option[value="' + window.jQuery.escapeSelector(val) + '"]').length) {
                            $sel.append(new Option(label || val, val, true, true));
                        }
                        $sel.val(val || null).trigger('change');
                    };

                    if (id) setVal(id, text || ('#' + id));
                    else if (text) setVal('name:' + text, text);
                    else setVal(null, '');
                    return;
                }

                const ensureOption = (val, label) => {
                    if (!val) return;
                    const exists = [...el.options].some(o => String(o.value) === String(val));
                    if (!exists) el.add(new Option(label || val, val, true, true));
                    el.value = val;
                };

                if (id) ensureOption(id, text || ('#' + id));
                else if (text) ensureOption('name:' + text, text);
                else el.value = '';
            }

            try {
                if (typeof window.__svcBuildOptions === 'function') {
                    if (wantSvcId) {
                        window.__svcBuildOptions(String(wantSvcId));
                    } else {
                        window.__svcBuildOptions({ byText: wantSvcText || '' });
                    }
                }
            } catch (_) {}

            selectServiceInDom(wantSvcId, wantSvcText);

            setTimeout(() => {
                selectServiceInDom(wantSvcId, wantSvcText);
            }, 60);

            pendingLocation = xp.location || '';
            setLocationSelectValue(pendingLocation || '');

            if (noteBody) noteBody.value = xp.description || '';

            const currentContact = (xp.contact || xp.email || xp.phone || '').trim();
            try {
                if (currentContact && typeof window.__setContactValue === 'function') {
                    window.__setContactValue(currentContact);
                }
            } catch (_) {}

            (async () => {
                let pid = Number(xp.participant_id || 0);
                if (!pid) {
                    try {
                        const titleNow = getTitleValue();
                        pid = await (window.__contacts?.guessPidFromTitle?.(titleNow) || Promise.resolve(null));
                    } catch (_) {}
                }
                window.currentParticipantId = pid || null;

                if (pid) {
                    try { await loadParticipantLocations(pid); } catch (_) {}
                    try { await window.__contacts?.loadForParticipant?.(pid); } catch (_) {}
                    try {
                        if (currentContact && typeof window.__setContactValue === 'function') {
                            window.__setContactValue(currentContact);
                        }
                    } catch (_) {}
                } else {
                    try { await window.__contacts?.loadForParticipant?.(null); } catch (_) {}
                }
            })();

            if (btnDel)  btnDel.style.display = 'inline-block';
            if (btnSave) btnSave.textContent  = 'Save';

            fetchSuggest(getTitleValue());
            updateQuickAddUI();
            (noteBody && (noteBody.value || '').trim()) ? showStep(2) : showStep(1);

            showModal();
        },

        eventDrop: ({ event, revert }) => {
            post(`${cfg.MOVE}/${encodeURIComponent(event.id)}.json`, {
                start: event.start ? event.start.toISOString().slice(0,19) : null,
                end:   event.end   ? event.end.toISOString().slice(0,19)   : null
            }).then(async ()=>{
                if (cfg.REM_RESYNC){ try{ await post(cfg.REM_RESYNC, { event_id:Number(event.id) }); }catch(_){ } }
            }).catch(()=>{ alert('Save failed'); revert(); });
        },

        eventResize: ({ event, revert }) => {
            post(`${cfg.MOVE}/${encodeURIComponent(event.id)}.json`, {
                start: event.start ? event.start.toISOString().slice(0,19) : null,
                end:   event.end   ? event.end.toISOString().slice(0,19)   : null
            }).then(async ()=>{
                if (cfg.REM_RESYNC){ try{ await post(cfg.REM_RESYNC, { event_id:Number(event.id) }); }catch(_){ } }
            }).catch(()=>{ alert('Save failed'); revert(); });
        },

        eventContent: function (arg) {
            const e = arg.event;
            const xp = e.extendedProps || {};
            const time = arg.timeText || '';
            const title= e.title || '';

            const typeBadge = xp.appointment_type
                ? `<span class="evt-badge type-${xp.appointment_type}">${xp.appointment_type === 'first' ? 'First' : 'Standard'}</span>` : '';

            const svcText = __resolveServiceLabel(xp);

            const serviceBadge = svcText
                ? `<span class="evt-badge svc-${esc(svcText.toLowerCase())}">
                        ${esc(formatServiceLabel(svcText))}
                   </span>`
                : '';

            const clinician = xp.clinician ? `<span class="evt-sub">${esc(xp.clinician)}</span>` : '';
            const location  = xp.location  ? `<span class="evt-sub">${esc(xp.location)}</span>` : '';
            const st = xp.status ? xp.status.charAt(0).toUpperCase()+xp.status.slice(1) : '';
            const statusBadge = xp.status ? `<span class="evt-badge st-${esc(xp.status)}">${esc(st)}</span>` : '';
            const hasNoteBadge = (xp.description || '').trim() ? `<span class="evt-badge note">Note</span>` : '';

            return { html: `<div class="evt"><span class="evt-time">${esc(time)}</span><span class="evt-title">${esc(title)}</span>${typeBadge}${serviceBadge}${clinician}${location}${statusBadge}${hasNoteBadge}</div>` };
        },
    });

    function openFlatpickrPicker(){
        const btn = document.querySelector('.fc-jumpDate-button');
        if (!__fp){
            const host = document.createElement('input');
            host.type = 'text'; host.id = 'fc-jump-date';
            Object.assign(host.style, { position:'fixed', top:'8px', left:'8px', width:'1px', height:'1px', opacity:'0', pointerEvents:'none', zIndex:1 });
            document.body.appendChild(host);

            __fp = flatpickr(host, {
                dateFormat:'Y-m-d',
                clickOpens:false,
                allowInput:false,
                appendTo: document.body,
                positionElement: btn || host,
                onOpen: function(){ setTimeout(()=>{ try{ __fp._input.blur(); }catch(_){} if (btn) btn.focus(); }, 0); },
                onChange: function(sel, dateStr){
                    if (!dateStr) return;
                    const sx = window.scrollX, sy = window.scrollY;
                    calendar.gotoDate(dateStr);
                    window.scrollTo(sx, sy);
                    setTimeout(()=>{ try{ __fp.close(); }catch(_){} if (btn) btn.focus(); }, 0);
                }
            });
        }
        try{ __fp.set('positionElement', btn); }catch(_){}
        __fp.setDate(calendar.getDate(), false);
        const sx = window.scrollX, sy = window.scrollY;
        __fp.open();
        setTimeout(()=>{ try{ __fp._input.blur(); }catch(_){} if (btn) btn.focus(); window.scrollTo(sx, sy); }, 0);
    }

    function openNativePicker(){
        let ip = document.getElementById('fc-jump-date-fallback');
        if (!ip){
            ip = document.createElement('input');
            ip.type='date'; ip.id='fc-jump-date-fallback';
            Object.assign(ip.style, { position:'fixed', top:'8px', left:'8px', opacity:'0', pointerEvents:'none', zIndex:1 });
            document.body.appendChild(ip);
            ip.addEventListener('change', function(){
                if (ip.value){
                    const sx = window.scrollX, sy = window.scrollY;
                    calendar.gotoDate(ip.value);
                    window.scrollTo(sx, sy);
                }
                setTimeout(()=>{ ip.value=''; ip.blur(); }, 0);
            });
        }
        const sx = window.scrollX, sy = window.scrollY;
        if (ip.showPicker) ip.showPicker(); else ip.click();
        setTimeout(()=>{ ip.blur(); window.scrollTo(sx, sy); }, 0);
    }

    calendar.render();

    window.__calRefetch = () => { try { calendar.refetchEvents(); } catch(_) {} };

    async function loadClinicians(){
        if (!cfg.CLIN) return;
        try{
            const res = await fetch(cfg.CLIN, { credentials:'same-origin' });
            const list= await res.json();

            const selClinEl = document.getElementById('clinician');
            const keepFilter = selClinEl ? selClinEl.value : '';
            const keepModal  = fClin  ? fClin.value  : '';

            if (selClinEl){
                selClinEl.innerHTML = '<option value="">All</option>';
                list.forEach((x)=> selClinEl.insertAdjacentHTML('beforeend', `<option value="${String(x.id)}">${esc(x.name)}</option>`));
                if (keepFilter && [...selClinEl.options].some(o=>o.value===keepFilter)) selClinEl.value = keepFilter;
            }
            if (fClin){
                fClin.innerHTML = '<option value="">Unassigned</option>';
                list.forEach((x)=> fClin.insertAdjacentHTML('beforeend', `<option value="${String(x.id)}">${esc(x.name)}</option>`));
                if (keepModal && [...fClin.options].some(o=>o.value===keepModal)) fClin.value = keepModal;
            }
            if (wlClinEl){
                wlClinEl.innerHTML = '<option value="">All clinicians</option>';
                list.forEach((x)=> wlClinEl.insertAdjacentHTML('beforeend', `<option value="${String(x.id)}">${esc(x.name)}</option>`));
            }
            if (pmClinSel){
                pmClinSel.innerHTML = '';
                list.forEach((x)=> pmClinSel.insertAdjacentHTML('beforeend', `<option value="${String(x.id)}">${esc(x.name)}</option>`));
                if ($ && $.fn.select2) {
                    const $dlg = $('#participantModal');
                    const $sel = $('#pmClinicians', $dlg);
                    try { $sel.select2('destroy'); } catch(e) {}
                    $sel.select2({ width:'100%', dropdownParent:$dlg, placeholder: $sel.data('placeholder') || '', allowClear:true, closeOnSelect:false });
                }
            }
        }catch(e){ console.warn('Clinicians load failed:', e); }
    }
    loadClinicians();

    if (selClin) selClin.addEventListener('change', ()=> calendar.refetchEvents());
    if (btnToday) btnToday.addEventListener('click', ()=> calendar.today());

    function bindTitleListeners(){
        const el = getTitleEl();
        if (!el) return;

        const debouncedResolve = debounce(async ()=>{
            await resolveParticipantIdByTitle();
            try { window.__contacts?.loadForParticipant(currentParticipantId || null); } catch(_){}
            updateQuickAddUI();
        }, 300);

        function maybeOpenQuickAddFromValue(){
            const v = getTitleValue();
            if (v && v.startsWith(ADD_OPT_PREFIX)) {
                const name = v.slice(ADD_OPT_PREFIX.length).trim();

                if (hasSelect2(el) && window.jQuery) {
                    const $el = window.jQuery(el);
                    $el.val(name).trigger('change.select2');
                } else {
                    el.value = name;
                }

                openQuickAddModal(name);

                fetchSuggest(name);
                updateQuickAddUI();

                return true;
            }
            return false;
        }

        el.__titleBound && el.__titleBound.forEach(({type,fn})=> el.removeEventListener(type, fn));
        el.__titleBound = [];

        if (hasSelect2(el) && window.jQuery) {
            const $el = window.jQuery(el);
            $el.off('change.__title input.__title select2:select.__title select2:clear.__title');

            const onChangeS2 = ()=>{
                if (maybeOpenQuickAddFromValue()) return;
                fetchSuggest(getTitleValue());
                debouncedResolve();
                updateQuickAddUI();
            };

            const onSelectS2 = ()=>{
                if (maybeOpenQuickAddFromValue()) return;
                fetchSuggest(getTitleValue());
                debouncedResolve();
                updateQuickAddUI();
            };

            const onClearS2 = ()=>{
                fetchSuggest('');
                debouncedResolve();
                updateQuickAddUI();
            };

            $el.on('change.__title', onChangeS2)
                .on('select2:select.__title', onSelectS2)
                .on('select2:clear.__title', onClearS2);

            el.__titleBound = [];
        } else {
            const onInput = (e)=>{
                if (maybeOpenQuickAddFromValue()) return;
                fetchSuggest(e.target.value.trim());
                debouncedResolve();
            };

            const onChange = async ()=>{
                if (maybeOpenQuickAddFromValue()) return;
                await resolveParticipantIdByTitle();
                try { window.__contacts?.loadForParticipant(currentParticipantId || null); } catch(_){}
                updateQuickAddUI();
            };

            el.addEventListener('input', onInput);
            el.addEventListener('change', onChange);
            el.__titleBound.push({type:'input', fn:onInput}, {type:'change', fn:onChange});
        }
    }
    bindTitleListeners();

    // function initTitleSelect2() {
    //     const el = getTitleEl();
    //     if (!el || !window.jQuery) return;
    //     const $el = window.jQuery(el);
    //     if ($el.data('select2')) $el.select2('destroy');
    //
    //     const nameOf = (x) => {
    //         const byParts = [x.first_name, x.middle_name, x.last_name].filter(Boolean).join(' ').trim();
    //         return (byParts || x.full_name || x.name || x.preferred || x.email || '').trim();
    //     };
    //
    //     const pick = (obj, paths) => {
    //         for (const p of paths) {
    //             const v = p.split('.').reduce((o,k)=> (o && o[k]!=null)? o[k] : undefined, obj);
    //             if (v !== undefined && v !== null && String(v).trim() !== '') return String(v).trim();
    //         }
    //         return '';
    //     };
    //
    //     const tryParseDate = (s) => {
    //         if (!s) return null;
    //         const str = String(s).trim();
    //
    //         let d = new Date(str);
    //         if (!isNaN(d)) return d;
    //
    //         let m = str.match(/^(\d{4})[\/\-\.](\d{1,2})[\/\-\.](\d{1,2})$/);
    //         if (m) { d = new Date(+m[1], +m[2]-1, +m[3]); if (!isNaN(d)) return d; }
    //
    //         m = str.match(/^(\d{1,2})[\/\-\.](\d{1,2})[\/\-\.](\d{4})$/);
    //         if (m) { d = new Date(+m[3], +m[2]-1, +m[1]); if (!isNaN(d)) return d; }
    //
    //         return null;
    //     };
    //
    //     const fmtDOB = (raw) => {
    //         if (!raw) return '';
    //         const d = tryParseDate(raw);
    //         if (!d) return String(raw); // 不认识就原样显示
    //         const mon = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'][d.getMonth()];
    //         const dd  = String(d.getDate()).padStart(2,'0');
    //         return `${dd} ${mon} ${d.getFullYear()}`;
    //     };
    //
    //     const calcAge = (raw) => {
    //         const d = tryParseDate(raw);
    //         if (!d) return '';
    //         const t = new Date();
    //         let age = t.getFullYear() - d.getFullYear();
    //         const m = t.getMonth() - d.getMonth();
    //         if (m < 0 || (m === 0 && t.getDate() < d.getDate())) age--;
    //         return (age >= 0 && age < 130) ? String(age) : '';
    //     };
    //
    //     const normGender = (v) => {
    //         const s = String(v || '').trim().toLowerCase();
    //         if (!s) return '';
    //         if (['m','male','man','boy'].includes(s)) return 'Male';
    //         if (['f','female','woman','girl'].includes(s)) return 'Female';
    //         if (['non-binary','nonbinary','nb','x','other','unknown','unspecified'].includes(s)) return 'Other';
    //         return s.charAt(0).toUpperCase() + s.slice(1);
    //     };
    //
    //     $el.select2({
    //         width: '100%',
    //         dropdownParent: window.jQuery('#createEventModal'),
    //         placeholder: el.getAttribute('placeholder') || 'Type a participant name…',
    //         allowClear: true,
    //         tags: true,
    //         minimumInputLength: 0,
    //
    //         ajax: (window.__CAL_CFG__ && window.__CAL_CFG__.PARTS) ? {
    //             delay: 200,
    //             transport: function (params, success, failure) {
    //                 const src = window.__CAL_CFG__.PARTS;
    //                 const term = String(params.data.term || '').trim();
    //                 const words = term.split(/\s+/).filter(Boolean);
    //                 const queries = words.length > 1 ? words : [term || ''];
    //
    //                 Promise.all(queries.map(q => {
    //                     const u = new URL(src, window.location.origin);
    //                     if (q) u.searchParams.set('q', q);
    //                     return fetch(u.toString(), { credentials: 'same-origin' })
    //                         .then(r => r.ok ? r.json() : [])
    //                         .catch(() => []);
    //                 }))
    //                     .then(pages => {
    //                         const seen = new Set(), merged = [];
    //                         pages.flat().forEach(x => {
    //                             const key = String(x.id ?? x.email ?? x.name ?? Math.random());
    //                             if (seen.has(key)) return; seen.add(key);
    //                             merged.push(x);
    //                         });
    //                         success({ results: merged });
    //                     })
    //                     .catch(failure);
    //             },
    //             processResults: function (data, params) {
    //                 const list = Array.isArray(data.results) ? data.results : [];
    //                 const term = String(params.term || '').trim().toLowerCase();
    //
    //                 const items = list.map(x => {
    //                     const name   = nameOf(x);
    //                     const dobRaw = pick(x, [
    //                         'dob','date_of_birth','birthday','birth_date','birthdate',
    //                         'profile.dob','profile.date_of_birth','profile.birthday',
    //                         'meta.dob','meta.date_of_birth','meta.birthday','DOB'
    //                     ]);
    //                     const genderRaw = pick(x, [
    //                         'gender','sex','profile.gender','profile.sex','meta.gender','meta.sex','Gender'
    //                     ]);
    //                     const email = pick(x, ['email','contact.email','contacts.email','profile.email']);
    //
    //                     return {
    //                         id: name,
    //                         text: name,
    //                         _email:  email,
    //                         _dob:    dobRaw,
    //                         _age:    calcAge(dobRaw),
    //                         _gender: normGender(genderRaw),
    //                         _raw:    x
    //                     };
    //                 });
    //
    //                 const hasExact = term && items.some(i => (i.text || '').toLowerCase() === term);
    //                 if (term && !hasExact) items.push({ id: term, text: `＋ Create "${params.term}"`, _isCreate: true });
    //
    //                 return { results: items };
    //             }
    //         } : null,
    //
    //         dropdownCssClass: 'dropdown-title',
    //         selectionCssClass: 'title-select',
    //
    //         templateResult: (di) => {
    //             if (!di || !di.id) return di?.text;
    //
    //             if (di._isCreate || di.id === '__add__' || di.element?.dataset?.add === '1') {
    //                 const wrap = document.createElement('div');
    //                 wrap.className = 'opt-add';
    //                 const main = di._isCreate ? di.text.replace(/^＋\s*Create\s*/,'Create ') : (di.text || '');
    //                 wrap.innerHTML = `
    //       <div class="plus">＋</div>
    //       <div>
    //         <div class="add-main">${main}</div>
    //         <div class="add-sub">Create a new participant</div>
    //       </div>`;
    //                 return wrap;
    //             }
    //
    //             const nameText = di.text || '';
    //             const subEmail = di._email || di.element?.dataset?.sub || '';
    //             const avatar   = (nameText.trim()[0] || '#').toUpperCase();
    //
    //             const dobStr = di._dob ? fmtDOB(di._dob) : '';
    //             const ageStr = di._age ? `${di._age}y` : '';
    //             const genStr = di._gender || '';
    //             const chips = [];
    //             if (dobStr || ageStr) chips.push(`${dobStr}${dobStr && ageStr ? ' • ' : ''}${ageStr}`);
    //             if (genStr) chips.push(genStr);
    //
    //             const wrap = document.createElement('div');
    //             wrap.className = 'opt-person';
    //             wrap.innerHTML = `
    //     <div class="opt-avatar">${avatar}</div>
    //     <div class="opt-text">
    //       <div class="opt-name">${nameText}</div>
    //       ${subEmail ? `<div class="opt-sub">${subEmail}</div>` : ''}
    //       ${chips.length ? `<div class="opt-meta">${chips.map(c => `<span class="chip">${c}</span>`).join(' ')}</div>` : ''}
    //     </div>`;
    //             return wrap;
    //         },
    //
    //         templateSelection: (di) => di? (di._isCreate ? String(di.id||'').trim() : (di.text||'')) : '',
    //
    //         createTag: (params) => {
    //             const term = (params.term || '').trim();
    //             return term ? { id: term, text: `＋ Create "${term}"`, _isCreate: true, newTag: true } : null;
    //         }
    //     });
    //
    //     $el.off('select2:select.__titleAdd').on('select2:select.__titleAdd', (e) => {
    //         const data = e.params?.data;
    //         if (data && (data._isCreate || data.id === '__add__' || data.element?.dataset?.add === '1')) {
    //             const name = String(data.id || '').trim() || getTitleValue();
    //             if (name) {
    //                 try {
    //                     const parts = name.split(/\s+/);
    //                     const first = parts.shift() || '';
    //                     const last  = parts.join(' ');
    //                     const pf = document.getElementById('pmFirst');
    //                     const pl = document.getElementById('pmLast');
    //                     if (pf) pf.value = first;
    //                     if (pl) pl.value = last;
    //                 } catch(_) {}
    //             }
    //             try { window.jQuery('#participantModal').modal('show'); } catch(_) {}
    //             $el.val(name || null).trigger('change.select2');
    //         }
    //     });
    // }

    if (tabBtnDetails) tabBtnDetails.addEventListener('click', (e)=>{ e.preventDefault(); showStep(1); });
    if (tabBtnNote)    tabBtnNote.addEventListener('click',    (e)=>{ e.preventDefault(); showStep(2); });
    if (btnNext) btnNext.addEventListener('click', ()=> showStep(2));
    if (btnBack) btnBack.addEventListener('click', ()=> showStep(1));

    if (noteTpl){
        noteTpl.addEventListener('change', ()=>{
            const key = noteTpl.value;
            const tpl = NOTE_TPLS[key] || '';
            if (!tpl || !noteBody) return;
            const current = noteBody.value || '';
            let action = 'replace';
            if (current.trim()){
                action = confirm('Replace current note with the selected template?\n\nClick "Cancel" to append at the end.')
                    ? 'replace' : 'append';
            }
            if (action==='replace') noteBody.value = tpl;
            else {
                const sep = current.endsWith('\n') ? '' : '\n\n';
                noteBody.value = current + sep + tpl;
            }
            const anchor='Subjective:'; const idx = noteBody.value.indexOf(anchor);
            const pos = idx>=0 ? idx+anchor.length+1 : noteBody.value.length;
            setTimeout(()=>{ try{ noteBody.focus(); noteBody.setSelectionRange(pos, pos);}catch(_){ } },0);
            showStep(2);
        });
    }

    if (btnSave){
        btnSave.addEventListener('click', function () {
            clearError();
            const titleNow = getTitleValue();
            try { if (window.__billingIsFixed && window.__billingIsFixed()) { if (fEnd) fEnd.value=''; } } catch(_){}
            if (!titleNow) return showError('Title is required.');
            if (!fDate.value || !fStart.value) return showError('Date/Start time is required.');
            const clinVal = fClin ? String(fClin.value || '').trim() : '';
            if (!clinVal) {
                showError('Clinician is required for new appointments.');
                try { fClin.focus(); } catch(_){}
                return;
            }
            function getServiceText() {
                const el = document.getElementById('evtService');
                if (!el) return '';
                const opt = el.selectedOptions && el.selectedOptions[0];
                return String(opt ? opt.textContent : '').trim();
            }

            const pickedSvc = (typeof window.__getSelectedService === 'function')
                ? window.__getSelectedService() : null;

            const svcText = getServiceText();
            const payload = {
                title: titleNow,
                appointment_type: fType ? fType.value : 'standard',
                service_type: svcText,
                service_name: svcText,
                service:      svcText,
                service_id: pickedSvc?.id ?? null,
                start: buildDateTime(fDate.value, fStart.value),
                end: fEnd.value ? buildDateTime(fDate.value, fEnd.value) : '',
                status: fStatus.value,
                user_id: fClin && fClin.value !== '' ? Number(fClin.value) : '',
                location: ($ && fLoc) ? String($(fLoc).val() || '').trim() : (fLoc?.value || '').trim(),
                description: (noteBody?.value || '').trim(),
                contact: (window.jQuery && window.jQuery.fn && window.jQuery.fn.select2)
                    ? String(window.jQuery('#evtContact').val() || '').trim()
                    : String(document.getElementById('evtContact')?.value || '').trim(),
                participant_id: (window.currentParticipantId || null)
            };

            const afterOk = async () => {
                if (chosenWaitId && cfg.WL_MARK){ try{ await post(`${cfg.WL_MARK}/${encodeURIComponent(chosenWaitId)}.json`, {}); }catch(_){ } }
                hideModal();
                calendar.refetchEvents();
            };

            if (mode === 'create'){
                post(cfg.STORE, payload)
                    .then(async (res)=>{
                        const newId = res && res.id;
                        try{
                            const to = (document.getElementById('evtContact')?.value || '').trim();
                            if (newId && to && cfg.REM_SEND_NOW){
                                await post(cfg.REM_SEND_NOW, { event_id: Number(newId), to_email: to, template: (fRemTpl?.value || 'default') });
                            }
                        }catch(e){
                            console.warn('Immediate email failed:', e);
                            if (errBox){ errBox.textContent = 'Email send failed (appointment was created).'; errBox.style.display = 'block'; }
                        }
                        try{
                            if (fRem && (fRem.value || '') === 'off'){
                                if (newId && cfg.REM_CANCEL) await post(cfg.REM_CANCEL, { event_id: Number(newId) });
                            }else if (newId && cfg.REM_SCHEDULE){
                                const val = (fRem.value || 'off').trim(); const offset = parseInt(val,10);
                                if (!Number.isNaN(offset)){
                                    await post(cfg.REM_SCHEDULE, { event_id: Number(newId), offset_minutes: offset, template: (fRemTpl?.value || 'default'), to_email: (document.getElementById('evtContact')?.value || '').trim() });
                                }
                            }
                        }catch(_){}
                        await afterOk();
                    })
                    .catch((err)=> showError(err.message || 'Create failed'));
            }else{
                post(`${cfg.UPDATE}/${encodeURIComponent(editingId)}.json`, payload)
                    .then(async ()=>{
                        try{
                            if (fRem && (fRem.value || '') === 'off'){ if (cfg.REM_CANCEL) await post(cfg.REM_CANCEL, { event_id: Number(editingId) }); }
                            else if (cfg.REM_SCHEDULE){
                                const val = (fRem.value || 'off').trim(); const offset = parseInt(val,10);
                                if (!Number.isNaN(offset)){
                                    await post(cfg.REM_SCHEDULE, { event_id: Number(editingId), offset_minutes: offset, template: (fRemTpl?.value || 'default'), to_email: (document.getElementById('evtContact')?.value || '').trim() });
                                }
                            }
                        }catch(_){}
                        await afterOk();
                    })
                    .catch((err)=> showError(err.message || 'Save failed'));
            }
        });
    }

    if (btnDel){
        btnDel.addEventListener('click', function(){
            if (!editingId) return;
            if (!confirm('Delete this event?')) return;
            post(`${cfg.DELETE}/${encodeURIComponent(editingId)}.json`, {})
                .then(()=>{ hideModal(); calendar.refetchEvents(); })
                .catch(()=>{ alert('Delete failed'); });
        });
    }

    if ($){
        $(document).on('hidden.bs.modal', '.modal', function () {
            if ($('.modal.show').length) $('body').addClass('modal-open');
        });
        $(document).on('hidden.bs.modal', '#waitListModal', function () {
            const $parent = $(modalId);
            if ($parent.length){
                $parent.trigger('focus');
                setTimeout(()=>{ $('#evtTitle').trigger('focus'); }, 0);
            }
        });
        $(document).on('shown.bs.modal', '#createEventModal', function(){
            bindTitleListeners();
        });
    }
})();
;(()=>{
    'use strict';
    if (window.__svc_addon_loaded) return;
    window.__svc_addon_loaded = true;

    const __cfg = window.__CAL_CFG__ || {};
    const $ = window.jQuery;

    const __svcPad = (n)=> (n<10?'0':'')+n;
    const __svcHM  = (d)=> `${__svcPad(d.getHours())}:${__svcPad(d.getMinutes())}`;
    const __svcAddMinutes=(d,mins)=>{ const t=new Date(d); t.setMinutes(t.getMinutes()+mins); return t; };
    const __svcCents = (v)=> Math.max(0, Math.round((parseFloat(v)||0)*100));
    const __svcMoney = (c,cur='AU$')=> `${cur}${(Number(c||0)/100).toFixed(2)}`;
    const __svcToJson = async (r)=>{ if(!r.ok) throw new Error(String(r.status)); return r.json(); };
    const __svcHasS2 = ($el)=> { try{ return !!$el && $el.hasClass('select2-hidden-accessible'); }catch(_){ return false; } };

    const __get = (id)=> document.getElementById(id);
    const __fDate  = ()=> __get('evtDate');
    const __fStart = ()=> __get('evtStart');
    const __fEnd   = ()=> __get('evtEnd');

    let __svcSelect = __get('evtService');

    function __svcFindAnchor(){
        return (__get('evtServiceType')?.closest('.form-group'))
            || (__get('evtType')?.closest('.form-group'))
            || (__get('evtDate')?.closest('.form-group'))
            || null;
    }

    function __svcInsertUI(){
        if (__svcSelect) return;
        const anchor = __svcFindAnchor();
        if (!anchor || !anchor.parentNode) return;
        const wrap = document.createElement('div');
        wrap.className = 'form-group';
        wrap.innerHTML = `
      <label for="evtService">Services Type</label>
      <select class="form-control" id="evtService"></select>
      <small class="form-text text-muted">Searchable; At the bottom, there is "+ New Service". After selection, the end time will be automatically filled in according to the service duration.</small>
    `;
        anchor.parentNode.insertBefore(wrap, anchor);
        __svcSelect = wrap.querySelector('#evtService');
    }

    const __LS_KEY='cal__services_v1';
    let __svcCache=[]; try{ __svcCache = JSON.parse(localStorage.getItem(__LS_KEY)||'[]')||[]; }catch(_){}
    const __svcNorm = (x)=>{
        if (!x) return null;
        const name = String(x.name || x.title || x.text || '').trim();
        if (!name) return null;
        return {
            id: x.id!=null ? Number(x.id) : null,
            name,
            duration_minutes: parseInt(x.duration_minutes ?? x.duration ?? 0,10) || 0,
            price_cents: x.price_cents!=null ? parseInt(x.price_cents,10)
                : x.price!=null       ? __svcCents(x.price) : 0
        };
    };
    const __svcSaveLocal = ()=>{ try{ localStorage.setItem(__LS_KEY, JSON.stringify(__svcCache)); }catch(_){} };

    async function __svcFetchServices(){
        const all = [];
        if (__cfg.ORG_SRV) {
            try {
                const r = await fetch(__cfg.ORG_SRV, { credentials:'same-origin' });
                if (!r.ok) {
                    const msg = await r.text().catch(()=>r.statusText);
                    console.warn('[Services] ORG_SRV failed:', r.status, msg);
                } else {
                    const rows = await r.json();
                    const orgItems = (Array.isArray(rows)?rows:[])
                        .map(__svcNorm)
                        .filter(Boolean)
                        .map(x => ({ ...x, group:'org' }));
                    all.push(...orgItems);
                }
            } catch (e) {
                console.warn('[Services] ORG_SRV error:', e);
            }
        }

        if (__cfg.SRV_FEED) {
            try {
                const u = new URL(__cfg.SRV_FEED, window.location.origin);
                const r = await fetch(u.toString(), { credentials:'same-origin' });
                if (!r.ok) {
                    const msg = await r.text().catch(()=>r.statusText);
                    console.warn('[Services] SRV_FEED failed:', r.status, msg);
                } else {
                    const rows = await r.json();
                    const sysItems = (Array.isArray(rows)?rows:[])
                        .map(__svcNorm)
                        .filter(Boolean)
                        .map(x => ({ ...x, group:'all' }));
                    all.push(...sysItems);
                }
            } catch (e) {
                console.warn('[Services] SRV_FEED error:', e, '→ using local cache');
                const cacheItems = (__svcCache||[]).map(x => ({...x, group:'all'}));
                all.push(...cacheItems);
            }
        }

        __svcCache = all.filter(x => x.group !== 'org');
        __svcSaveLocal();

        return all;
    }


    async function __svcCreateService(payload){
        if (!__cfg.SRV_ADD){
            const fake = { id: Date.now(), ...payload };
            __svcCache.push(fake); __svcSaveLocal();
            return { ok:true, service: fake, fallback:true };
        }
        try{
            const res = await fetch(__cfg.SRV_ADD, {
                method:'POST', credentials:'same-origin',
                headers:{'Content-Type':'application/json','X-CSRF-Token': (__cfg.CSRF||'')},
                body: JSON.stringify(payload)
            }).then(__svcToJson);
            const s = __svcNorm(res.service||res||{});
            if (s){
                const i = __svcCache.findIndex(it=>it.id===s.id);
                if (i>=0) __svcCache[i]=s; else __svcCache.push(s);
                __svcSaveLocal();
                return { ok:true, service:s };
            }
            throw new Error('Create failed');
        }catch(_){
            const fake = { id: Date.now(), ...payload };
            __svcCache.push(fake); __svcSaveLocal();
            return { ok:true, service: fake, fallback:true };
        }
    }

    function __svcHandleSelectChange(optEl){
        const val = optEl?.value || '';
        if (!val || val==='__add__'){ window.__svcSelected = null; return; }
        const id  = val.startsWith('name:') ? null : Number(val);
        const name= optEl.textContent.trim();
        const dur = parseInt(optEl.dataset.duration||'0',10)||0;
        const price=parseInt(optEl.dataset.price||'0',10)||0;
        window.__svcSelected = { id, name, duration_minutes:dur, price_cents:price };

        try { if (window.__billingIsFixed && window.__billingIsFixed()) return; } catch(_){}

        const fD=__fDate(), fS=__fStart(), fE=__fEnd();
        if (dur && fD && fS && fE && fD.value && fS.value){
            const st = new Date(`${fD.value}T${fS.value}:00`);
            fE.value = __svcHM(__svcAddMinutes(st, dur));
        }
    }

    async function __svcBuildOptions(selectPref){
        __svcInsertUI();
        if (!__svcSelect) return;

        const list = await __svcFetchServices();

        const seenById = new Set();
        const seenByName = new Set();
        const merged = [];
        for (const s of list) {
            const keyId = s.id != null ? `id:${s.id}` : null;
            const keyName = `name:${(s.name||'').toLowerCase()}`;
            if (keyId && seenById.has(keyId)) continue;
            if (seenByName.has(keyName)) continue;
            if (keyId) seenById.add(keyId);
            seenByName.add(keyName);
            merged.push(s);
        }

        if ($ && __svcHasS2($( __svcSelect ))) { try{ $(__svcSelect).select2('destroy'); }catch(_){ } }
        __svcSelect.innerHTML = '';
        __svcSelect.appendChild(new Option('', '', true, false));

        const ogOrg = document.createElement('optgroup');
        ogOrg.label = "Organisation's Service";
        let orgCount = 0;

        merged
            .sort((a,b)=>{
                if ((a.group==='org') !== (b.group==='org')) return a.group==='org' ? -1 : 1;
                return (a.name||'').localeCompare(b.name||'');
            })
            .forEach(s=>{
                const value = (s.group === 'org')
                    ? `name:${s.name}`
                    : (s.id != null ? String(s.id) : `name:${s.name}`);

                const opt = new Option(s.name, value, false, false);
                opt.dataset.duration = String(s.duration_minutes || 0);
                opt.dataset.price    = String(s.price_cents || 0);
                if (s.group === 'org')
                {
                    opt.dataset.duration = String(s.duration_minutes || 0);
                    opt.dataset.price    = String(s.price_cents || 0);
                    ogOrg.appendChild(opt);
                    orgCount++;
                }else
                {
                    opt.dataset.duration = String(s.duration_minutes || 0);
                    opt.dataset.price    = String(s.price_cents || 0);
                    __svcSelect.appendChild(opt);
                }

            });

        if (orgCount > 0) {
            __svcSelect.insertBefore(ogOrg, __svcSelect.firstChild.nextSibling); // 第一个是空占位
        }

        __svcSelect.appendChild(new Option('＋ New Service', '__add__', false, false));

        const selectId   = (typeof selectPref === 'string' || typeof selectPref === 'number') ? String(selectPref) : null;
        const selectText = (selectPref && typeof selectPref === 'object') ? String(selectPref.byText||'') : '';

        if ($ && $.fn && $.fn.select2){
            $(__svcSelect).select2({
                width:'100%',
                dropdownParent: $('#createEventModal'),
                allowClear:true,
                placeholder:'Choose The Service',
                templateResult: (di)=>{
                    if (!di.id) return di.text;
                    if (di.id==='__add__'){ const el=document.createElement('div'); el.innerHTML='<strong>＋ New Service</strong>'; return el; }
                    const opt  = di.element;
                    const dur  = opt?.dataset?.duration || '0';
                    const price= opt?.dataset?.price || '0';
                    const el   = document.createElement('div');
                    el.innerHTML = `<div style="font-size:14px;">${di.text||''}</div>
                        <div class="text-muted" style="font-size:12px;">${dur} Minutes ・ ${__svcMoney(price)}</div>`;
                    return el;
                }
            });

            $(__svcSelect).off('select2:select.__svc').on('select2:select.__svc', (e)=>{
                if (e.params.data.id==='__add__'){ __svcOpenCreateModal(); $(__svcSelect).val(null).trigger('change'); return; }
                __svcHandleSelectChange(e.params.data.element);
            });

            if (selectId != null){
                if ($( __svcSelect ).find(`option[value="${selectId.replace(/"/g,'\\"')}"]`).length){
                    $( __svcSelect ).val(selectId).trigger('change');
                }
            } else if (selectText) {
                let matched = false, matchedRow = null;
                $(__svcSelect).find('option').each(function(){
                    if (this.textContent.trim() === selectText){
                        $(__svcSelect).val(this.value);
                        matched = true;
                        return false;
                    }
                });
                if (!matched) {
                    matchedRow = (window.__svcAll || []).find(x => x.name === selectText) || null;

                    const opt = new Option(selectText, 'name:'+selectText, true, true);
                    opt.dataset.duration = String(matchedRow?.duration_minutes || 0);
                    opt.dataset.price    = String(matchedRow?.price_cents || 0);
                    __svcSelect.appendChild(opt);
                }
                $(__svcSelect).trigger('change');
            } else {
                const keep = $(__svcSelect).val();
                $(__svcSelect).val(keep || null).trigger('change');
            }
        } else {
            __svcSelect.removeEventListener('change', __svcBuildOptions.__onChange);
            __svcBuildOptions.__onChange = function(){
                const el = __svcSelect.options[__svcSelect.selectedIndex];
                if (!el) return;
                if (el.value === '__add__'){ __svcOpenCreateModal(); __svcSelect.value=''; return; }
                __svcHandleSelectChange(el);
            };
            __svcSelect.addEventListener('change', __svcBuildOptions.__onChange);

            if (selectId != null) {
                __svcSelect.value = selectId;
            } else if (selectText) {
                let matched = false;
                [...__svcSelect.options].forEach(o=>{ if (o.textContent.trim() === selectText){ __svcSelect.value = o.value; matched = true; } });
                if (!matched){
                    __svcSelect.add(new Option(selectText, 'name:'+selectText, true, true));
                    __svcSelect.value = 'name:'+selectText;
                }
            }
            const el = __svcSelect.options[__svcSelect.selectedIndex];
            if (el && el.value && el.value !== '__add__') __svcHandleSelectChange(el);
        }
        window.__svcAll = merged;
        window.__svcNameMap = window.__svcNameMap || {};
        merged.forEach(x => { if (x && x.id != null && x.name) window.__svcNameMap[String(x.id)] = String(x.name); });
        try {
            const cache = JSON.parse(localStorage.getItem('cal__services_v1') || '[]') || [];
            const byId = Object.fromEntries(cache.map(x => [String(x.id), x]));
            merged.forEach(x => { if (x && x.id != null) byId[String(x.id)] = x; });
            localStorage.setItem('cal__services_v1', JSON.stringify(Object.values(byId)));
        } catch(_){}

        if (window.__calRefetch) window.__calRefetch();
    }

    function __svcEnsureModal(){
        if (document.getElementById('serviceModal')) return;
        const html = `
<div class="modal fade" id="serviceModal" tabindex="-1" role="dialog" aria-hidden="true">
  <div class="modal-dialog modal-xl" role="document">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add new services</h5>
        <button type="button" class="close" data-dismiss="modal" aria-label="Close"><span>×</span></button>
      </div>
      <div class="modal-body">
        <form id="srvForm" novalidate>
          <div class="form-group">
            <label>Name</label>
            <input type="text" class="form-control" id="srvName" required>
          </div>
          <div class="form-group">
            <label>Code</label>
            <input type="text" class="form-control" id="srvCode" placeholder="e.g. A001">
          </div>
          <div class="form-row">
            <div class="form-group col-md-6">
              <label>Duration (minutes)</label>
              <input type="number" class="form-control" id="srvDur" min="0" step="5" value="45">
            </div>
            <div class="form-group col-md-6">
              <label>Price (AU$)</label>
              <input type="number" class="form-control" id="srvPrice" min="0" step="0.01" value="100">
            </div>
          </div>
        </form>
        <div class="text-danger small" id="srvErr" style="display:none;"></div>
      </div>
      <div class="modal-footer">
        <button type="button" id="btnSrvCreate" class="btn btn-primary">Create</button>
        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
      </div>
    </div>
  </div>
</div>`;
        document.body.insertAdjacentHTML('beforeend', html);

        if ($){
            $('#btnSrvCreate').on('click', async ()=>{
                const $err = $('#srvErr'); $err.hide().text('');
                const name = String($('#srvName').val()||'').trim();
                const code = String($('#srvCode').val()||'').trim();
                const dur  = parseInt(String($('#srvDur').val()||'0'),10)||0;
                const price= __svcCents($('#srvPrice').val());
                if (!name){ $err.text('Name is required').show(); return; }

                const r = await __svcCreateService({ name, code, duration_minutes:dur, price_cents:price });
                if (!r || !r.ok){ $err.text('Create failed').show(); return; }
                $('#serviceModal').modal('hide');
                await __svcBuildOptions(r.service?.id);
            });
        }
    }

    function __svcOpenCreateModal(){ __svcEnsureModal(); if ($) $('#serviceModal').modal('show'); }

    if ($){
        $(document).on('shown.bs.modal', '#createEventModal', function(){
            if (window.__calMode === 'edit') return;
            __svcBuildOptions();
        });
    }
    __svcInsertUI();
    __svcBuildOptions();

    Object.defineProperty(window, '__getSelectedService', { value: ()=> window.__svcSelected || null });
    Object.defineProperty(window, '__svcBuildOptions', { value: __svcBuildOptions });

    function getContactEl() {
        return document.getElementById('evtContact');
    }

    function setContactValue(val) {
        const el = getContactEl();
        if (!el) return;
        const txt = String(val || '').trim();
        if (window.jQuery && window.jQuery.fn && window.jQuery.fn.select2) {
            const $sel = window.jQuery(el);
            if (txt && !$sel.find('option[value="' + txt.replace(/"/g, '\\"') + '"]').length) {
                $sel.append(new Option(txt, txt, true, true));
            }
            $sel.val(txt || '').trigger('change');
        } else {
            if (txt) {
                let exists = Array.from(el.options || []).some(o => o.value === txt);
                if (!exists) el.add(new Option(txt, txt, true, true));
                el.value = txt;
            } else {
                el.value = '';
            }
        }
    }

    window.__setContactValue   = setContactValue;

})();
;(()=>{
    'use strict';
    if (window.__contact_addon_loaded) return;
    window.__contact_addon_loaded = true;

    const cfg = window.__CAL_CFG__ || {};
    const $   = window.jQuery;

    function ensureSelect(){
        const el = document.getElementById('evtContact');
        if (!el) return null;
        if (el.tagName && el.tagName.toLowerCase() === 'select') return el;

        const sel = document.createElement('select');
        sel.className = el.className || 'form-control';
        sel.id        = el.id || 'evtContact';
        if (el.placeholder) sel.dataset.placeholder = el.placeholder;
        el.parentNode.replaceChild(sel, el);
        return sel;
    }

    function setContactValue(val){
        const sel = ensureSelect();
        if (!sel) return;
        const v = String(val || '').trim();

        if ($ && $.fn && $.fn.select2) {
            const $sel = $(sel);
            if (v && !$sel.find('option[value="'+v.replace(/"/g,'\\"')+'"]').length) {
                $sel.append(new Option(v, v, true, true));
            }
            $sel.val(v || '').trigger('change');
        } else {
            if (v) {
                let exists = Array.from(sel.options || []).some(o => o.value === v);
                if (!exists) sel.add(new Option(v, v, true, true));
                sel.value = v;
            } else {
                sel.value = '';
            }
        }
    }
    function getSelectedContact(){
        const sel = document.getElementById('evtContact');
        if (!sel) return '';
        if ($ && $.fn && $.fn.select2) {
            return String($(sel).val() || '').trim();
        }
        return String(sel.value || '').trim();
    }

    async function fetchContacts(pid){
        if (!pid || !cfg.PART_CONTACTS) return [];
        const url = `${cfg.PART_CONTACTS}?participant_id=${encodeURIComponent(pid)}`;
        try{
            const r = await fetch(url, { credentials: 'same-origin' });
            if (!r.ok) throw new Error(String(r.status));
            const list = await r.json();
            return Array.isArray(list) ? list : [];
        }catch(e){
            console.warn('contacts load failed', e);
            return [];
        }
    }

    let contactSel = ensureSelect();

    function initSelect2IfNeeded(sel){
        if (!($ && $.fn && $.fn.select2)) return;

        const $dlg = $('#createEventModal');
        const $sel = $(sel);

        if ($sel.data('select2')) $sel.select2('destroy');

        $sel.select2({
            width: '100%',
            dropdownParent: $dlg.length ? $dlg : undefined,
            placeholder: $sel.data('placeholder') || 'Email or phone…',
            allowClear: true,
            tags: true,
            createTag: (params)=> ({ id: params.term, text: params.term, isNew: true }),
            templateResult: (di)=>{
                if (!di.id) return di.text;
                const el = di.element;
                const type    = el?.dataset?.type || '';
                const primary = el?.dataset?.primary === '1';
                const n = document.createElement('div');
                n.innerHTML =
                    `<div style="display:flex;align-items:center;gap:8px;">
             <span>${di.text||''}</span>
             ${type ? `<span class="badge badge-light">${type}</span>` : ''}
             ${primary ? '<span class="badge badge-primary">Primary</span>' : ''}
           </div>`;
                return n;
            },
            templateSelection: (di)=>{
                if (!di.id) return di.text;
                const el = di.element;
                const primary = el?.dataset?.primary === '1';
                const n = document.createElement('div');
                n.innerHTML =
                    `<div style="display:flex;align-items:center;gap:6px;">
             <span>${di.text||''}</span>
             ${primary ? '<span class="badge badge-primary">Primary</span>' : ''}
           </div>`;
                return n;
            }
        });
    }

    async function buildOptions(pid, keepValue){
        contactSel = ensureSelect();
        if (!contactSel) return;

        const list = await fetchContacts(pid);

        const seen = new Set();
        const rows = [];
        for (const c of list) {
            const k = (c.type||'') + '|' + String(c.value||'').toLowerCase();
            if (seen.has(k)) continue;
            seen.add(k);
            rows.push({
                type: (c.type||'').toLowerCase(),
                value: String(c.value||'').trim(),
                is_primary: c.is_primary ? 1 : 0
            });
        }
        rows.sort((a,b)=>{
            if (a.is_primary !== b.is_primary) return b.is_primary - a.is_primary;
            if (a.type !== b.type) return a.type === 'email' ? -1 : 1;
            return a.value.localeCompare(b.value);
        });

        contactSel.innerHTML = '';
        contactSel.appendChild(new Option('', '', true, false));
        rows.forEach(c=>{
            if (!c.value) return;
            const opt = new Option(c.value, c.value, false, false);
            opt.dataset.type    = c.type || '';
            opt.dataset.primary = c.is_primary ? '1' : '';
            contactSel.appendChild(opt);
        });

        initSelect2IfNeeded(contactSel);

        if (keepValue) setContactValue(keepValue);
        else if (rows.length && rows[0].is_primary) setContactValue(rows[0].value);
    }

    async function loadForParticipant(pid){
        const keep = getSelectedContact();
        await buildOptions(pid, keep);
    }

    async function guessPidFromTitle(title){
        const url = cfg.PARTS ? `${cfg.PARTS}?q=${encodeURIComponent(title||'')}` : null;
        if (!url) return null;
        try{
            const r = await fetch(url, { credentials:'same-origin' });
            const list = await r.json();
            const t = String(title||'').trim().toLowerCase();
            const hit = Array.isArray(list) ? list.find(x => String(x.name||x.title||'').trim().toLowerCase()===t) : null;
            return (hit && hit.id) ? hit.id : (Array.isArray(list) && list[0]?.id) || null;
        }catch(_){ return null; }
    }

    window.__contacts = { loadForParticipant, guessPidFromTitle };
    window.__getSelectedContact= getSelectedContact;

    if ($) {
        $(document).on('shown.bs.modal', '#createEventModal', function(){
            contactSel = ensureSelect();
            initSelect2IfNeeded(contactSel);
        });
    }

})();
